<?php

namespace App\Console\Commands;

use App\Jobs\SendInvoiceReminder;
use App\Models\Invoice;
use Illuminate\Console\Command;

class SendInvoiceReminders extends Command
{
    protected $signature = 'invoices:send-reminders';
    protected $description = 'Send reminders for overdue and due invoices';

    public function handle(): int
    {
        $this->info('Checking for invoices requiring reminders...');

        // Get overdue invoices
        $overdueInvoices = Invoice::whereIn('status', ['sent', 'viewed', 'partial'])
            ->where('due_date', '<', now())
            ->where(function ($q) {
                $q->whereNull('last_reminder_at')
                  ->orWhere('last_reminder_at', '<', now()->subDays(7));
            })
            ->get();

        $this->info("Found {$overdueInvoices->count()} overdue invoices");

        foreach ($overdueInvoices as $invoice) {
            SendInvoiceReminder::dispatch($invoice, 'overdue');
            $this->line("  - Queued reminder for invoice {$invoice->invoice_number}");
        }

        // Get invoices due soon (3 days)
        $dueSoonInvoices = Invoice::whereIn('status', ['sent', 'viewed', 'partial'])
            ->whereBetween('due_date', [now(), now()->addDays(3)])
            ->where(function ($q) {
                $q->whereNull('last_reminder_at')
                  ->orWhere('last_reminder_at', '<', now()->subDays(3));
            })
            ->get();

        $this->info("Found {$dueSoonInvoices->count()} invoices due soon");

        foreach ($dueSoonInvoices as $invoice) {
            SendInvoiceReminder::dispatch($invoice, 'due_soon');
            $this->line("  - Queued reminder for invoice {$invoice->invoice_number}");
        }

        $this->info('Done!');

        return Command::SUCCESS;
    }
}
