<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Attendance;
use App\Models\Employee;
use Illuminate\Http\Request;

class AttendanceController extends Controller
{
    public function index(Request $request)
    {
        $query = Attendance::where('tenant_id', $request->user()->tenant_id)
            ->with('employee.user');

        if ($request->employee_id) {
            $query->where('employee_id', $request->employee_id);
        }

        if ($request->month && $request->year) {
            $query->whereMonth('date', $request->month)
                ->whereYear('date', $request->year);
        }

        $attendances = $query->orderByDesc('date')->paginate($request->per_page ?? 20);

        $stats = [
            'total_days' => $attendances->total(),
            'present' => Attendance::where('tenant_id', $request->user()->tenant_id)
                ->where('status', 'present')->count(),
            'late' => Attendance::where('tenant_id', $request->user()->tenant_id)
                ->where('status', 'late')->count(),
            'absent' => Attendance::where('tenant_id', $request->user()->tenant_id)
                ->where('status', 'absent')->count(),
        ];

        return response()->json([
            'data' => $attendances->items(),
            'stats' => $stats,
            'meta' => [
                'total' => $attendances->total(),
                'per_page' => $attendances->perPage(),
                'current_page' => $attendances->currentPage(),
            ],
        ]);
    }

    public function checkIn(Request $request)
    {
        $employee = Employee::where('user_id', $request->user()->id)->firstOrFail();

        $existing = Attendance::where('employee_id', $employee->id)
            ->where('date', now()->toDateString())
            ->first();

        if ($existing) {
            return response()->json(['message' => 'Already checked in today'], 422);
        }

        $checkIn = $request->input('check_in', now()->format('H:i:s'));
        $status = $checkIn > '09:00:00' ? 'late' : 'present';

        $attendance = Attendance::create([
            'tenant_id' => $request->user()->tenant_id,
            'employee_id' => $employee->id,
            'date' => now()->toDateString(),
            'check_in' => $checkIn,
            'status' => $status,
            'ip_address' => $request->ip(),
        ]);

        return response()->json(['data' => $attendance]);
    }

    public function checkOut(Request $request)
    {
        $employee = Employee::where('user_id', $request->user()->id)->firstOrFail();

        $attendance = Attendance::where('employee_id', $employee->id)
            ->where('date', now()->toDateString())
            ->whereNull('check_out')
            ->firstOrFail();

        $attendance->update([
            'check_out' => now()->format('H:i:s'),
            'hours_worked' => $attendance->calculateHoursWorked(),
        ]);

        return response()->json(['data' => $attendance]);
    }

    public function today(Request $request)
    {
        $employee = Employee::where('user_id', $request->user()->id)->first();

        if (!$employee) {
            return response()->json(['data' => null]);
        }

        $attendance = Attendance::where('employee_id', $employee->id)
            ->where('date', now()->toDateString())
            ->first();

        return response()->json(['data' => $attendance]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'date' => 'required|date',
            'check_in' => 'required',
            'check_out' => 'nullable',
            'status' => 'required|in:present,late,absent,half_day',
            'notes' => 'nullable|string',
        ]);

        $attendance = Attendance::create([
            'tenant_id' => $request->user()->tenant_id,
            ...$validated,
        ]);

        return response()->json(['data' => $attendance], 201);
    }
}
