<?php

namespace App\Http\Controllers\Api;

use App\Models\BankAccount;
use App\Models\BankTransaction;
use App\Services\BankingService;
use App\Jobs\SyncBankTransactions;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class BankAccountController extends ApiController
{
    public function __construct(protected BankingService $bankingService) {}

    public function index(Request $request): JsonResponse
    {
        $accounts = BankAccount::where('tenant_id', $request->user()->tenant_id)
            ->withSum('transactions as pending_transactions', fn($q) => $q->where('status', 'pending'))
            ->orderBy('name')
            ->get();

        return $this->success($accounts);
    }

    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'account_number' => 'required|string|max:50',
            'bank_name' => 'required|string|max:255',
            'account_type' => 'required|in:checking,savings,credit_card',
            'currency' => 'required|string|size:3',
            'opening_balance' => 'nullable|numeric',
            'chart_account_id' => 'nullable|exists:chart_of_accounts,id',
        ]);

        $account = BankAccount::create([
            'tenant_id' => $request->user()->tenant_id,
            ...$validated,
            'current_balance' => $validated['opening_balance'] ?? 0,
        ]);

        return $this->success($account, 'Bank account created', 201);
    }

    public function show(BankAccount $account): JsonResponse
    {
        $this->authorize('view', $account);
        return $this->success($account->load('chartAccount'));
    }

    public function update(Request $request, BankAccount $account): JsonResponse
    {
        $this->authorize('update', $account);

        $validated = $request->validate([
            'name' => 'sometimes|string|max:255',
            'bank_name' => 'sometimes|string|max:255',
            'chart_account_id' => 'nullable|exists:chart_of_accounts,id',
            'is_active' => 'boolean',
        ]);

        $account->update($validated);
        return $this->success($account, 'Bank account updated');
    }

    public function destroy(BankAccount $account): JsonResponse
    {
        $this->authorize('delete', $account);

        if ($account->transactions()->exists()) {
            return $this->error('Cannot delete account with transactions', 422);
        }

        $account->delete();
        return $this->success(null, 'Bank account deleted');
    }

    public function sync(BankAccount $account): JsonResponse
    {
        $this->authorize('update', $account);

        if (!$account->plaid_access_token) {
            return $this->error('Account not connected to Plaid', 422);
        }

        SyncBankTransactions::dispatch($account);
        return $this->success(null, 'Sync started');
    }

    public function transactions(Request $request): JsonResponse
    {
        $query = BankTransaction::whereHas('bankAccount', fn($q) => 
            $q->where('tenant_id', $request->user()->tenant_id)
        )->with(['bankAccount:id,name', 'account:id,name']);

        if ($request->bank_account_id) {
            $query->where('bank_account_id', $request->bank_account_id);
        }

        if ($request->status) {
            $query->where('status', $request->status);
        }

        if ($request->from_date) {
            $query->where('transaction_date', '>=', $request->from_date);
        }

        if ($request->to_date) {
            $query->where('transaction_date', '<=', $request->to_date);
        }

        $transactions = $query->orderBy('transaction_date', 'desc')
            ->paginate($request->per_page ?? 50);

        return $this->paginated($transactions);
    }

    public function updateTransaction(Request $request, BankTransaction $transaction): JsonResponse
    {
        $validated = $request->validate([
            'account_id' => 'nullable|exists:chart_of_accounts,id',
            'contact_id' => 'nullable|exists:contacts,id',
            'notes' => 'nullable|string',
        ]);

        $transaction->update($validated);
        return $this->success($transaction, 'Transaction updated');
    }

    public function matchTransaction(Request $request, BankTransaction $transaction): JsonResponse
    {
        $validated = $request->validate([
            'match_type' => 'required|in:invoice,bill,account',
            'match_id' => 'required|integer',
        ]);

        try {
            match ($validated['match_type']) {
                'invoice' => $this->bankingService->matchToInvoice($transaction, $validated['match_id']),
                'bill' => $this->bankingService->matchToBill($transaction, $validated['match_id']),
                'account' => $this->bankingService->matchTransaction($transaction, $validated['match_id']),
            };

            return $this->success($transaction->fresh(), 'Transaction matched');
        } catch (\Exception $e) {
            return $this->error($e->getMessage(), 422);
        }
    }

    public function suggestions(BankTransaction $transaction): JsonResponse
    {
        $suggestions = $this->bankingService->getUnmatchedSuggestions($transaction);
        return $this->success($suggestions);
    }
}
