<?php

namespace App\Http\Controllers\Api;

use App\Models\Bill;
use App\Models\BillPayment;
use Illuminate\Http\Request;

class BillController extends ApiController
{
    public function index(Request $request)
    {
        $query = Bill::where('tenant_id', $request->user()->tenant_id)
            ->with(['vendor', 'items']);

        if ($request->status) {
            $query->where('status', $request->status);
        }

        if ($request->vendor_id) {
            $query->where('vendor_id', $request->vendor_id);
        }

        if ($request->from_date) {
            $query->whereDate('bill_date', '>=', $request->from_date);
        }

        if ($request->to_date) {
            $query->whereDate('bill_date', '<=', $request->to_date);
        }

        if ($request->search) {
            $query->where(function ($q) use ($request) {
                $q->where('bill_number', 'like', "%{$request->search}%")
                  ->orWhere('vendor_invoice_number', 'like', "%{$request->search}%")
                  ->orWhereHas('vendor', fn($vq) => $vq->where('name', 'like', "%{$request->search}%"));
            });
        }

        $bills = $query->orderBy('bill_date', 'desc')
            ->paginate($request->per_page ?? 15);

        return $this->paginated($bills);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'vendor_id' => 'required|exists:vendors,id',
            'vendor_invoice_number' => 'nullable|string|max:50',
            'bill_date' => 'required|date',
            'due_date' => 'required|date|after_or_equal:bill_date',
            'currency' => 'nullable|string|size:3',
            'items' => 'required|array|min:1',
            'items.*.name' => 'required|string|max:255',
            'items.*.description' => 'nullable|string',
            'items.*.quantity' => 'required|numeric|min:0.01',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.account_id' => 'nullable|exists:chart_of_accounts,id',
            'items.*.tax_rate_id' => 'nullable|exists:tax_rates,id',
            'discount_type' => 'nullable|in:percentage,fixed',
            'discount_value' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string',
            'terms' => 'nullable|string',
        ]);

        $bill = Bill::create([
            'tenant_id' => $request->user()->tenant_id,
            'vendor_id' => $validated['vendor_id'],
            'vendor_invoice_number' => $validated['vendor_invoice_number'] ?? null,
            'bill_date' => $validated['bill_date'],
            'due_date' => $validated['due_date'],
            'currency' => $validated['currency'] ?? 'BHD',
            'discount_type' => $validated['discount_type'] ?? null,
            'discount_value' => $validated['discount_value'] ?? 0,
            'notes' => $validated['notes'] ?? null,
            'terms' => $validated['terms'] ?? null,
            'created_by' => $request->user()->id,
        ]);

        foreach ($validated['items'] as $index => $item) {
            $bill->items()->create([
                'tenant_id' => $request->user()->tenant_id,
                'name' => $item['name'],
                'description' => $item['description'] ?? null,
                'quantity' => $item['quantity'],
                'unit_price' => $item['unit_price'],
                'account_id' => $item['account_id'] ?? null,
                'tax_rate_id' => $item['tax_rate_id'] ?? null,
                'subtotal' => $item['quantity'] * $item['unit_price'],
                'position' => $index,
            ]);
        }

        $bill->calculateTotals();
        $bill->save();

        return $this->success($bill->load('items', 'vendor'), 'Bill created successfully', 201);
    }

    public function show(Bill $bill)
    {
        return $this->success($bill->load('items', 'vendor', 'payments', 'createdBy'));
    }

    public function update(Request $request, Bill $bill)
    {
        if ($bill->status !== Bill::STATUS_DRAFT) {
            return $this->error('Only draft bills can be edited', 400);
        }

        $validated = $request->validate([
            'vendor_id' => 'sometimes|exists:vendors,id',
            'vendor_invoice_number' => 'nullable|string|max:50',
            'bill_date' => 'sometimes|date',
            'due_date' => 'sometimes|date',
            'items' => 'sometimes|array|min:1',
            'items.*.name' => 'required|string|max:255',
            'items.*.quantity' => 'required|numeric|min:0.01',
            'items.*.unit_price' => 'required|numeric|min:0',
            'discount_type' => 'nullable|in:percentage,fixed',
            'discount_value' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string',
        ]);

        $bill->update($validated);

        if (isset($validated['items'])) {
            $bill->items()->delete();
            
            foreach ($validated['items'] as $index => $item) {
                $bill->items()->create([
                    'tenant_id' => $bill->tenant_id,
                    'name' => $item['name'],
                    'description' => $item['description'] ?? null,
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                    'account_id' => $item['account_id'] ?? null,
                    'tax_rate_id' => $item['tax_rate_id'] ?? null,
                    'subtotal' => $item['quantity'] * $item['unit_price'],
                    'position' => $index,
                ]);
            }

            $bill->calculateTotals();
            $bill->save();
        }

        return $this->success($bill->load('items'), 'Bill updated successfully');
    }

    public function destroy(Bill $bill)
    {
        if ($bill->status !== Bill::STATUS_DRAFT) {
            return $this->error('Only draft bills can be deleted', 400);
        }

        $bill->items()->delete();
        $bill->delete();

        return $this->success(null, 'Bill deleted successfully');
    }

    public function recordPayment(Request $request, Bill $bill)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:0.01|max:' . $bill->getAmountDue(),
            'payment_date' => 'required|date',
            'payment_method' => 'required|in:cash,bank_transfer,credit_card,cheque',
            'bank_account_id' => 'nullable|exists:bank_accounts,id',
            'reference' => 'nullable|string|max:100',
            'description' => 'nullable|string',
        ]);

        $payment = BillPayment::create([
            'tenant_id' => $bill->tenant_id,
            'bill_id' => $bill->id,
            'vendor_id' => $bill->vendor_id,
            'bank_account_id' => $validated['bank_account_id'] ?? null,
            'amount' => $validated['amount'],
            'payment_date' => $validated['payment_date'],
            'payment_method' => $validated['payment_method'],
            'reference' => $validated['reference'] ?? null,
            'description' => $validated['description'] ?? null,
            'status' => 'completed',
            'created_by' => $request->user()->id,
        ]);

        $bill->updatePaymentStatus();

        return $this->success($payment, 'Payment recorded successfully', 201);
    }

    public function markReceived(Bill $bill)
    {
        $bill->markAsReceived();
        return $this->success($bill, 'Bill marked as received');
    }

    public function void(Bill $bill)
    {
        if ($bill->amount_paid > 0) {
            return $this->error('Cannot void a bill with payments', 400);
        }

        $bill->void();
        return $this->success($bill, 'Bill voided successfully');
    }

    public function duplicate(Bill $bill)
    {
        $newBill = $bill->replicate(['bill_number', 'status', 'amount_paid']);
        $newBill->status = Bill::STATUS_DRAFT;
        $newBill->bill_date = now();
        $newBill->due_date = now()->addDays(30);
        $newBill->save();

        foreach ($bill->items as $item) {
            $newBill->items()->create($item->only([
                'tenant_id', 'name', 'description', 'quantity', 'unit_price',
                'account_id', 'tax_rate_id', 'subtotal', 'total', 'position'
            ]));
        }

        return $this->success($newBill->load('items'), 'Bill duplicated successfully', 201);
    }

    public function summary(Request $request)
    {
        $tenantId = $request->user()->tenant_id;

        $totalBilled = Bill::where('tenant_id', $tenantId)->sum('total');
        $totalPaid = Bill::where('tenant_id', $tenantId)->sum('amount_paid');
        $outstanding = $totalBilled - $totalPaid;
        
        $overdueBills = Bill::where('tenant_id', $tenantId)
            ->where('due_date', '<', now())
            ->whereColumn('amount_paid', '<', 'total')
            ->get();

        return $this->success([
            'total_billed' => $totalBilled,
            'total_paid' => $totalPaid,
            'outstanding' => $outstanding,
            'overdue_count' => $overdueBills->count(),
            'overdue_amount' => $overdueBills->sum('total') - $overdueBills->sum('amount_paid'),
        ]);
    }

    public function aging(Request $request)
    {
        $tenantId = $request->user()->tenant_id;
        $today = now();

        $aging = [
            'current' => 0,
            '1_30' => 0,
            '31_60' => 0,
            '61_90' => 0,
            'over_90' => 0,
        ];

        $bills = Bill::where('tenant_id', $tenantId)
            ->whereColumn('amount_paid', '<', 'total')
            ->get();

        foreach ($bills as $bill) {
            $amountDue = $bill->getAmountDue();
            $daysOverdue = $today->diffInDays($bill->due_date, false);

            if ($daysOverdue <= 0) {
                $aging['current'] += $amountDue;
            } elseif ($daysOverdue <= 30) {
                $aging['1_30'] += $amountDue;
            } elseif ($daysOverdue <= 60) {
                $aging['31_60'] += $amountDue;
            } elseif ($daysOverdue <= 90) {
                $aging['61_90'] += $amountDue;
            } else {
                $aging['over_90'] += $amountDue;
            }
        }

        return $this->success($aging);
    }

    public function overdue(Request $request)
    {
        $bills = Bill::where('tenant_id', $request->user()->tenant_id)
            ->where('due_date', '<', now())
            ->whereColumn('amount_paid', '<', 'total')
            ->with('vendor')
            ->orderBy('due_date')
            ->get();

        return $this->success($bills);
    }
}
