<?php

namespace App\Http\Controllers\Api;

use App\Models\Campaign;
use App\Services\CampaignService;
use App\Http\Requests\StoreCampaignRequest;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class CampaignController extends ApiController
{
    public function __construct(protected CampaignService $campaignService) {}

    public function index(Request $request): JsonResponse
    {
        $campaigns = Campaign::where('tenant_id', $request->user()->tenant_id)
            ->when($request->type, fn($q, $type) => $q->where('type', $type))
            ->when($request->status, fn($q, $status) => $q->where('status', $status))
            ->orderBy('created_at', 'desc')
            ->paginate($request->per_page ?? 20);

        return $this->paginated($campaigns);
    }

    public function store(StoreCampaignRequest $request): JsonResponse
    {
        $campaign = $this->campaignService->create(
            $request->user()->tenant_id,
            $request->validated(),
            $request->user()->id
        );

        return $this->success($campaign, 'Campaign created', 201);
    }

    public function show(Campaign $campaign): JsonResponse
    {
        $this->authorize('view', $campaign);
        return $this->success($campaign->load(['recipients' => fn($q) => $q->limit(100)]));
    }

    public function update(Request $request, Campaign $campaign): JsonResponse
    {
        $this->authorize('update', $campaign);

        if ($campaign->status !== Campaign::STATUS_DRAFT) {
            return $this->error('Cannot update non-draft campaign', 422);
        }

        $campaign->update($request->validated());
        return $this->success($campaign, 'Campaign updated');
    }

    public function destroy(Campaign $campaign): JsonResponse
    {
        $this->authorize('delete', $campaign);
        $campaign->delete();
        return $this->success(null, 'Campaign deleted');
    }

    public function send(Campaign $campaign): JsonResponse
    {
        $this->authorize('update', $campaign);

        try {
            $this->campaignService->send($campaign);
            return $this->success($campaign->fresh(), 'Campaign queued for sending');
        } catch (\Exception $e) {
            return $this->error($e->getMessage(), 422);
        }
    }

    public function schedule(Request $request, Campaign $campaign): JsonResponse
    {
        $this->authorize('update', $campaign);

        $request->validate(['scheduled_at' => 'required|date|after:now']);

        $this->campaignService->schedule($campaign, new \DateTime($request->scheduled_at));
        return $this->success($campaign->fresh(), 'Campaign scheduled');
    }

    public function pause(Campaign $campaign): JsonResponse
    {
        $this->authorize('update', $campaign);

        if ($campaign->status !== Campaign::STATUS_SENDING) {
            return $this->error('Campaign is not currently sending', 422);
        }

        $this->campaignService->pause($campaign);
        return $this->success($campaign->fresh(), 'Campaign paused');
    }

    public function resume(Campaign $campaign): JsonResponse
    {
        $this->authorize('update', $campaign);

        if ($campaign->status !== Campaign::STATUS_PAUSED) {
            return $this->error('Campaign is not paused', 422);
        }

        $this->campaignService->resume($campaign);
        return $this->success($campaign->fresh(), 'Campaign resumed');
    }

    public function stats(Campaign $campaign): JsonResponse
    {
        $this->authorize('view', $campaign);

        $stats = [
            'total_recipients' => $campaign->recipient_count,
            'sent' => $campaign->recipients()->where('status', 'sent')->count(),
            'delivered' => $campaign->recipients()->where('status', 'delivered')->count(),
            'opened' => $campaign->opened_count,
            'clicked' => $campaign->clicked_count,
            'bounced' => $campaign->bounced_count,
            'unsubscribed' => $campaign->recipients()->where('status', 'unsubscribed')->count(),
            'open_rate' => $campaign->sent_count > 0 
                ? round(($campaign->opened_count / $campaign->sent_count) * 100, 2) : 0,
            'click_rate' => $campaign->opened_count > 0 
                ? round(($campaign->clicked_count / $campaign->opened_count) * 100, 2) : 0,
        ];

        return $this->success($stats);
    }

    public function duplicate(Campaign $campaign): JsonResponse
    {
        $this->authorize('view', $campaign);

        $newCampaign = $this->campaignService->duplicate($campaign, request()->user()->id);
        return $this->success($newCampaign, 'Campaign duplicated', 201);
    }
}
