<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreDepartmentRequest;
use App\Models\Department;
use App\Services\DepartmentService;
use Illuminate\Http\Request;

class DepartmentController extends Controller
{
    public function __construct(protected DepartmentService $service) {}

    public function index(Request $request)
    {
        $departments = Department::where('tenant_id', $request->user()->tenant_id)
            ->with(['manager.user', 'parent'])
            ->withCount('employees')
            ->orderBy('name')
            ->get();

        return response()->json(['data' => $departments]);
    }

    public function store(StoreDepartmentRequest $request)
    {
        $department = Department::create([
            'tenant_id' => $request->user()->tenant_id,
            ...$request->validated(),
        ]);

        return response()->json(['data' => $department->load('manager.user')], 201);
    }

    public function show(Department $department)
    {
        $this->authorize('view', $department);

        return response()->json([
            'data' => $department->load(['manager.user', 'employees.user', 'children']),
        ]);
    }

    public function update(Request $request, Department $department)
    {
        $this->authorize('update', $department);

        $department->update($request->validate([
            'name' => 'sometimes|string|max:255',
            'code' => 'nullable|string|max:20',
            'description' => 'nullable|string|max:500',
            'manager_id' => 'nullable|exists:employees,id',
            'parent_id' => 'nullable|exists:departments,id',
            'budget' => 'nullable|numeric|min:0',
        ]));

        return response()->json(['data' => $department->load('manager.user')]);
    }

    public function destroy(Department $department)
    {
        $this->authorize('delete', $department);

        if ($department->employees()->exists()) {
            return response()->json([
                'message' => 'Cannot delete department with employees',
            ], 422);
        }

        $department->delete();

        return response()->noContent();
    }

    public function hierarchy(Request $request)
    {
        $departments = Department::where('tenant_id', $request->user()->tenant_id)
            ->whereNull('parent_id')
            ->with(['children.children', 'manager.user'])
            ->withCount('employees')
            ->get();

        return response()->json(['data' => $departments]);
    }

    public function stats(Request $request)
    {
        $stats = $this->service->getStats($request->user()->tenant_id);
        return response()->json(['data' => $stats]);
    }
}
