<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;

class DeployController extends Controller
{
    /**
     * Secret key for deployment (set in .env as DEPLOY_SECRET)
     */
    public function deploy(Request $request)
    {
        // Verify deploy secret
        $secret = $request->header('X-Deploy-Secret');
        if ($secret !== config('app.deploy_secret')) {
            return response()->json(['error' => 'Unauthorized'], 401);
        }

        $results = [];

        try {
            // Pull latest code if using Git
            if (File::exists(base_path('.git'))) {
                exec('cd ' . base_path() . ' && git pull origin main 2>&1', $output);
                $results['git_pull'] = implode("\n", $output);
            }

            // Run migrations
            Artisan::call('migrate', ['--force' => true]);
            $results['migrate'] = Artisan::output();

            // Clear and cache
            Artisan::call('config:cache');
            Artisan::call('route:cache');
            Artisan::call('view:cache');
            $results['cache'] = 'Caches cleared and rebuilt';

            // Restart queue if supervisor is available
            exec('supervisorctl restart laravel-worker 2>&1', $queueOutput);
            $results['queue'] = implode("\n", $queueOutput);

            Log::info('Deployment successful', $results);

            return response()->json([
                'success' => true,
                'message' => 'Deployment completed successfully',
                'results' => $results,
                'timestamp' => now()->toIso8601String(),
            ]);

        } catch (\Exception $e) {
            Log::error('Deployment failed: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Receive and apply file updates
     */
    public function updateFile(Request $request)
    {
        $secret = $request->header('X-Deploy-Secret');
        if ($secret !== config('app.deploy_secret')) {
            return response()->json(['error' => 'Unauthorized'], 401);
        }

        $validated = $request->validate([
            'path' => 'required|string',
            'content' => 'required|string',
            'backup' => 'boolean',
        ]);

        $fullPath = base_path($validated['path']);

        // Create backup if requested
        if ($request->input('backup', true) && File::exists($fullPath)) {
            File::copy($fullPath, $fullPath . '.backup.' . time());
        }

        // Ensure directory exists
        File::ensureDirectoryExists(dirname($fullPath));

        // Write file
        File::put($fullPath, base64_decode($validated['content']));

        Log::info('File updated via deploy API: ' . $validated['path']);

        return response()->json([
            'success' => true,
            'path' => $validated['path'],
            'message' => 'File updated successfully',
        ]);
    }

    /**
     * Get current version/status
     */
    public function status(Request $request)
    {
        $secret = $request->header('X-Deploy-Secret');
        if ($secret !== config('app.deploy_secret')) {
            return response()->json(['error' => 'Unauthorized'], 401);
        }

        return response()->json([
            'app_name' => config('app.name'),
            'environment' => config('app.env'),
            'php_version' => PHP_VERSION,
            'laravel_version' => app()->version(),
            'last_migration' => $this->getLastMigration(),
            'disk_free' => disk_free_space('/'),
            'timestamp' => now()->toIso8601String(),
        ]);
    }

    private function getLastMigration()
    {
        try {
            return \DB::table('migrations')->orderBy('id', 'desc')->first()->migration ?? 'unknown';
        } catch (\Exception $e) {
            return 'error';
        }
    }
}
