<?php

namespace App\Http\Controllers\Api;

use App\Models\Estimate;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;

class EstimateController extends ApiController
{
    public function index(Request $request)
    {
        $query = Estimate::where('tenant_id', $request->user()->tenant_id)
            ->with(['contact', 'items']);

        if ($request->status) {
            $query->where('status', $request->status);
        }

        if ($request->contact_id) {
            $query->where('contact_id', $request->contact_id);
        }

        if ($request->from_date) {
            $query->whereDate('estimate_date', '>=', $request->from_date);
        }

        if ($request->to_date) {
            $query->whereDate('estimate_date', '<=', $request->to_date);
        }

        if ($request->search) {
            $query->where(function ($q) use ($request) {
                $q->where('estimate_number', 'like', "%{$request->search}%")
                  ->orWhereHas('contact', fn($cq) => $cq->where('first_name', 'like', "%{$request->search}%")
                      ->orWhere('last_name', 'like', "%{$request->search}%")
                      ->orWhere('company_name', 'like', "%{$request->search}%"));
            });
        }

        $estimates = $query->orderBy('estimate_date', 'desc')
            ->paginate($request->per_page ?? 15);

        return $this->paginated($estimates);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'contact_id' => 'required|exists:contacts,id',
            'reference' => 'nullable|string|max:100',
            'estimate_date' => 'required|date',
            'expiry_date' => 'nullable|date|after_or_equal:estimate_date',
            'currency' => 'nullable|string|size:3',
            'billing_address' => 'nullable|array',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'nullable|exists:products,id',
            'items.*.name' => 'required|string|max:255',
            'items.*.description' => 'nullable|string',
            'items.*.quantity' => 'required|numeric|min:0.01',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.tax_rate_id' => 'nullable|exists:tax_rates,id',
            'discount_type' => 'nullable|in:percentage,fixed',
            'discount_value' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string',
            'terms' => 'nullable|string',
        ]);

        $estimate = Estimate::create([
            'tenant_id' => $request->user()->tenant_id,
            'contact_id' => $validated['contact_id'],
            'reference' => $validated['reference'] ?? null,
            'estimate_date' => $validated['estimate_date'],
            'expiry_date' => $validated['expiry_date'] ?? now()->addDays(30),
            'currency' => $validated['currency'] ?? 'BHD',
            'billing_address' => $validated['billing_address'] ?? null,
            'discount_type' => $validated['discount_type'] ?? null,
            'discount_value' => $validated['discount_value'] ?? 0,
            'notes' => $validated['notes'] ?? null,
            'terms' => $validated['terms'] ?? null,
            'created_by' => $request->user()->id,
        ]);

        foreach ($validated['items'] as $index => $item) {
            $estimate->items()->create([
                'tenant_id' => $request->user()->tenant_id,
                'product_id' => $item['product_id'] ?? null,
                'name' => $item['name'],
                'description' => $item['description'] ?? null,
                'quantity' => $item['quantity'],
                'unit_price' => $item['unit_price'],
                'tax_rate_id' => $item['tax_rate_id'] ?? null,
                'subtotal' => $item['quantity'] * $item['unit_price'],
                'position' => $index,
            ]);
        }

        $estimate->calculateTotals();
        $estimate->save();

        return $this->success($estimate->load('items', 'contact'), 'Estimate created successfully', 201);
    }

    public function show(Estimate $estimate)
    {
        return $this->success($estimate->load('items', 'contact', 'createdBy', 'convertedInvoice'));
    }

    public function update(Request $request, Estimate $estimate)
    {
        if ($estimate->status === Estimate::STATUS_CONVERTED) {
            return $this->error('Cannot edit converted estimates', 400);
        }

        $validated = $request->validate([
            'contact_id' => 'sometimes|exists:contacts,id',
            'reference' => 'nullable|string|max:100',
            'estimate_date' => 'sometimes|date',
            'expiry_date' => 'nullable|date',
            'items' => 'sometimes|array|min:1',
            'items.*.name' => 'required|string|max:255',
            'items.*.quantity' => 'required|numeric|min:0.01',
            'items.*.unit_price' => 'required|numeric|min:0',
            'discount_type' => 'nullable|in:percentage,fixed',
            'discount_value' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string',
            'terms' => 'nullable|string',
        ]);

        $estimate->update($validated);

        if (isset($validated['items'])) {
            $estimate->items()->delete();
            
            foreach ($validated['items'] as $index => $item) {
                $estimate->items()->create([
                    'tenant_id' => $estimate->tenant_id,
                    'product_id' => $item['product_id'] ?? null,
                    'name' => $item['name'],
                    'description' => $item['description'] ?? null,
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                    'tax_rate_id' => $item['tax_rate_id'] ?? null,
                    'subtotal' => $item['quantity'] * $item['unit_price'],
                    'position' => $index,
                ]);
            }

            $estimate->calculateTotals();
            $estimate->save();
        }

        return $this->success($estimate->load('items'), 'Estimate updated successfully');
    }

    public function destroy(Estimate $estimate)
    {
        if ($estimate->status === Estimate::STATUS_CONVERTED) {
            return $this->error('Cannot delete converted estimates', 400);
        }

        $estimate->items()->delete();
        $estimate->delete();

        return $this->success(null, 'Estimate deleted successfully');
    }

    public function send(Request $request, Estimate $estimate)
    {
        $validated = $request->validate([
            'to' => 'required|email',
            'cc' => 'nullable|array',
            'cc.*' => 'email',
            'message' => 'nullable|string',
        ]);

        // Send email with PDF attachment
        // Mail::to($validated['to'])->send(new EstimateMail($estimate, $validated['message']));

        $estimate->markAsSent();

        return $this->success($estimate, 'Estimate sent successfully');
    }

    public function pdf(Estimate $estimate)
    {
        $tenant = $estimate->tenant;
        
        $pdf = Pdf::loadView('estimates.pdf', [
            'estimate' => $estimate->load('items', 'contact'),
            'tenant' => $tenant,
        ]);

        return $pdf->download("estimate-{$estimate->estimate_number}.pdf");
    }

    public function accept(Estimate $estimate)
    {
        if (!in_array($estimate->status, [Estimate::STATUS_SENT, Estimate::STATUS_VIEWED])) {
            return $this->error('Estimate cannot be accepted', 400);
        }

        $estimate->accept();

        return $this->success($estimate, 'Estimate accepted');
    }

    public function decline(Request $request, Estimate $estimate)
    {
        $validated = $request->validate([
            'reason' => 'nullable|string|max:500',
        ]);

        $estimate->decline($validated['reason'] ?? null);

        return $this->success($estimate, 'Estimate declined');
    }

    public function convertToInvoice(Estimate $estimate)
    {
        if ($estimate->status !== Estimate::STATUS_ACCEPTED) {
            return $this->error('Only accepted estimates can be converted', 400);
        }

        $invoice = $estimate->convertToInvoice();

        return $this->success([
            'estimate' => $estimate,
            'invoice' => $invoice->load('items'),
        ], 'Estimate converted to invoice successfully');
    }

    public function duplicate(Estimate $estimate)
    {
        $newEstimate = $estimate->replicate(['estimate_number', 'status', 'sent_at', 'viewed_at', 'accepted_at', 'declined_at', 'converted_to_invoice_id']);
        $newEstimate->status = Estimate::STATUS_DRAFT;
        $newEstimate->estimate_date = now();
        $newEstimate->expiry_date = now()->addDays(30);
        $newEstimate->save();

        foreach ($estimate->items as $item) {
            $newEstimate->items()->create($item->only([
                'tenant_id', 'product_id', 'name', 'description', 'quantity',
                'unit_price', 'tax_rate_id', 'subtotal', 'total', 'position'
            ]));
        }

        return $this->success($newEstimate->load('items'), 'Estimate duplicated successfully', 201);
    }
}
