<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreFormSubmissionRequest;
use App\Http\Resources\FormResource;
use App\Http\Resources\FormSubmissionResource;
use App\Models\Form;
use App\Services\FormService;
use Illuminate\Http\Request;

class FormController extends Controller
{
    public function __construct(protected FormService $service) {}

    public function index(Request $request)
    {
        $forms = Form::where('tenant_id', $request->user()->tenant_id)
            ->withCount('submissions')
            ->orderByDesc('created_at')
            ->paginate($request->per_page ?? 20);

        return FormResource::collection($forms);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string|max:1000',
            'fields' => 'required|array|min:1',
            'fields.*.label' => 'required|string|max:100',
            'fields.*.name' => 'required|string|max:50',
            'fields.*.type' => 'required|in:text,email,phone,number,textarea,select,radio,checkbox,date',
            'fields.*.required' => 'boolean',
            'fields.*.options' => 'nullable|string',
            'thank_you_message' => 'nullable|string|max:500',
            'redirect_url' => 'nullable|url',
            'is_active' => 'boolean',
        ]);

        $form = $this->service->create($request->user()->tenant_id, $validated);

        return new FormResource($form);
    }

    public function show(Form $form)
    {
        $this->authorize('view', $form);
        
        return new FormResource($form->loadCount('submissions'));
    }

    public function update(Request $request, Form $form)
    {
        $this->authorize('update', $form);
        
        $form->update($request->validated());
        
        return new FormResource($form);
    }

    public function destroy(Form $form)
    {
        $this->authorize('delete', $form);
        
        $form->delete();
        
        return response()->noContent();
    }

    public function showBySlug(string $slug)
    {
        $form = Form::where('slug', $slug)
            ->where('is_active', true)
            ->firstOrFail();

        return new FormResource($form);
    }

    public function submit(string $slug, StoreFormSubmissionRequest $request)
    {
        $form = Form::where('slug', $slug)
            ->where('is_active', true)
            ->firstOrFail();

        $submission = $this->service->submit(
            $form,
            $request->validated(),
            $request->ip()
        );

        return response()->json([
            'message' => $form->thank_you_message,
            'redirect_url' => $form->redirect_url,
        ]);
    }

    public function submissions(Form $form)
    {
        $this->authorize('view', $form);
        
        $submissions = $form->submissions()
            ->with('contact')
            ->orderByDesc('created_at')
            ->paginate(20);

        return FormSubmissionResource::collection($submissions);
    }

    public function duplicate(Form $form)
    {
        $this->authorize('update', $form);
        
        $newForm = $this->service->duplicate($form);
        
        return new FormResource($newForm);
    }
}
