<?php

namespace App\Http\Controllers\Api;

use App\Jobs\ExportData;
use App\Jobs\ImportContacts;
use App\Models\Export;
use App\Models\Import;
use App\Services\ExportService;
use App\Services\ImportService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Storage;

class ImportExportController extends ApiController
{
    public function __construct(
        protected ImportService $importService,
        protected ExportService $exportService
    ) {}

    public function uploadForImport(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'file' => 'required|file|mimes:csv,txt|max:10240',
            'type' => 'required|in:contacts,products,invoices',
        ]);

        $path = $request->file('file')->store('imports/' . $request->user()->tenant_id);

        $preview = $this->importService->getPreview($path);
        $suggestedMapping = $this->importService->getSuggestedMapping($preview['headers']);

        return $this->success([
            'file_path' => $path,
            'headers' => $preview['headers'],
            'preview' => $preview['preview'],
            'total_rows' => $preview['total_rows'],
            'suggested_mapping' => $suggestedMapping,
        ]);
    }

    public function startImport(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'file_path' => 'required|string',
            'type' => 'required|in:contacts,products,invoices',
            'mapping' => 'required|array',
        ]);

        if (!Storage::exists($validated['file_path'])) {
            return $this->error('File not found', 404);
        }

        ImportContacts::dispatch(
            $request->user()->tenant_id,
            $validated['file_path'],
            $validated['mapping'],
            $request->user()->id
        );

        return $this->success(['status' => 'processing'], 'Import started');
    }

    public function importStatus(Request $request, Import $import): JsonResponse
    {
        $this->authorize('view', $import);
        return $this->success($import);
    }

    public function imports(Request $request): JsonResponse
    {
        $imports = Import::where('tenant_id', $request->user()->tenant_id)
            ->orderByDesc('created_at')
            ->paginate($request->get('per_page', 15));

        return $this->paginated($imports);
    }

    public function startExport(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'type' => 'required|in:contacts,invoices,bills,payments,tasks',
            'filters' => 'nullable|array',
        ]);

        ExportData::dispatch(
            $request->user()->tenant_id,
            $validated['type'],
            $validated['filters'] ?? [],
            $request->user()->id
        );

        return $this->success(['status' => 'processing'], 'Export started');
    }

    public function exportStatus(Request $request, Export $export): JsonResponse
    {
        $this->authorize('view', $export);
        
        $data = $export->toArray();
        
        if ($export->status === 'completed' && $export->file_path) {
            $data['download_url'] = $this->exportService->getDownloadUrl($export);
        }

        return $this->success($data);
    }

    public function exports(Request $request): JsonResponse
    {
        $exports = Export::where('tenant_id', $request->user()->tenant_id)
            ->orderByDesc('created_at')
            ->paginate($request->get('per_page', 15));

        return $this->paginated($exports);
    }

    public function downloadExport(Request $request, Export $export): JsonResponse
    {
        $this->authorize('view', $export);

        if ($export->status !== 'completed' || !$export->file_path) {
            return $this->error('Export not ready', 422);
        }

        return $this->success([
            'download_url' => $this->exportService->getDownloadUrl($export)
        ]);
    }
}
