<?php

namespace App\Http\Controllers\Api;

use App\Models\Invoice;
use App\Models\InvoiceItem;
use App\Models\Payment;
use App\Models\JournalEntry;
use App\Services\InvoiceService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Barryvdh\DomPDF\Facade\Pdf;

class InvoiceController extends ApiController
{
    public function __construct(protected InvoiceService $invoiceService) {}

    public function index(Request $request): JsonResponse
    {
        $query = Invoice::with(['contact', 'createdBy'])
            ->when($request->status, fn($q, $s) => $q->where('status', $s))
            ->when($request->contact_id, fn($q, $id) => $q->where('contact_id', $id))
            ->when($request->from_date, fn($q, $d) => $q->whereDate('invoice_date', '>=', $d))
            ->when($request->to_date, fn($q, $d) => $q->whereDate('invoice_date', '<=', $d))
            ->when($request->search, fn($q, $s) => 
                $q->where('invoice_number', 'like', "%{$s}%")
                  ->orWhereHas('contact', fn($c) => $c->where('first_name', 'like', "%{$s}%"))
            )
            ->orderBy($request->sort_by ?? 'invoice_date', $request->sort_dir ?? 'desc');

        return $this->paginated($query->paginate($request->per_page ?? 25));
    }

    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'contact_id' => 'required|exists:contacts,id',
            'opportunity_id' => 'nullable|exists:opportunities,id',
            'invoice_date' => 'required|date',
            'due_date' => 'required|date|after_or_equal:invoice_date',
            'currency' => 'nullable|string|size:3',
            'notes' => 'nullable|string',
            'terms' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.name' => 'required|string|max:255',
            'items.*.description' => 'nullable|string',
            'items.*.quantity' => 'required|numeric|min:0.01',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.tax_rate_id' => 'nullable|exists:tax_rates,id',
            'items.*.discount' => 'nullable|numeric|min:0',
        ]);

        $invoice = $this->invoiceService->create($validated);

        return $this->success($invoice->load(['contact', 'items']), 'Invoice created', 201);
    }

    public function show(Invoice $invoice): JsonResponse
    {
        return $this->success($invoice->load(['contact', 'items', 'payments', 'createdBy']));
    }

    public function update(Request $request, Invoice $invoice): JsonResponse
    {
        if ($invoice->status !== 'draft') {
            return $this->error('Cannot update sent invoice', 422);
        }

        $validated = $request->validate([
            'contact_id' => 'sometimes|exists:contacts,id',
            'invoice_date' => 'sometimes|date',
            'due_date' => 'sometimes|date',
            'notes' => 'nullable|string',
            'terms' => 'nullable|string',
            'items' => 'sometimes|array|min:1',
            'items.*.id' => 'nullable|exists:invoice_items,id',
            'items.*.name' => 'required|string|max:255',
            'items.*.quantity' => 'required|numeric|min:0.01',
            'items.*.unit_price' => 'required|numeric|min:0',
        ]);

        $invoice = $this->invoiceService->update($invoice, $validated);

        return $this->success($invoice->load(['contact', 'items']), 'Invoice updated');
    }

    public function destroy(Invoice $invoice): JsonResponse
    {
        if ($invoice->status !== 'draft') {
            return $this->error('Cannot delete sent invoice', 422);
        }

        $invoice->items()->delete();
        $invoice->delete();

        return $this->success(null, 'Invoice deleted');
    }

    public function send(Invoice $invoice): JsonResponse
    {
        $this->invoiceService->send($invoice);
        return $this->success($invoice->fresh(), 'Invoice sent');
    }

    public function markPaid(Request $request, Invoice $invoice): JsonResponse
    {
        $request->validate([
            'amount' => 'required|numeric|min:0.01',
            'payment_date' => 'required|date',
            'method' => 'required|in:cash,bank_transfer,credit_card,cheque,paypal,stripe,other',
            'reference' => 'nullable|string',
        ]);

        $payment = $this->invoiceService->recordPayment($invoice, $request->all());

        return $this->success([
            'invoice' => $invoice->fresh(),
            'payment' => $payment,
        ], 'Payment recorded');
    }

    public function pdf(Invoice $invoice)
    {
        $pdf = Pdf::loadView('invoices.pdf', [
            'invoice' => $invoice->load(['contact', 'items', 'tenant']),
        ]);

        return $pdf->download("invoice-{$invoice->invoice_number}.pdf");
    }

    public function duplicate(Invoice $invoice): JsonResponse
    {
        $newInvoice = $this->invoiceService->duplicate($invoice);
        return $this->success($newInvoice->load(['contact', 'items']), 'Invoice duplicated', 201);
    }

    public function void(Invoice $invoice): JsonResponse
    {
        if ($invoice->status === 'cancelled') {
            return $this->error('Invoice already cancelled', 422);
        }

        $this->invoiceService->void($invoice);
        return $this->success($invoice->fresh(), 'Invoice voided');
    }

    public function summary(Request $request): JsonResponse
    {
        $from = $request->from_date ?? now()->startOfMonth();
        $to = $request->to_date ?? now()->endOfMonth();

        $summary = [
            'total_invoiced' => Invoice::whereBetween('invoice_date', [$from, $to])->sum('total'),
            'total_paid' => Invoice::whereBetween('invoice_date', [$from, $to])->sum('amount_paid'),
            'total_outstanding' => Invoice::whereIn('status', ['sent', 'viewed', 'partial', 'overdue'])->sum('amount_due'),
            'overdue_count' => Invoice::where('status', 'overdue')->count(),
            'overdue_amount' => Invoice::where('status', 'overdue')->sum('amount_due'),
            'by_status' => Invoice::selectRaw('status, COUNT(*) as count, SUM(total) as total')
                ->groupBy('status')->get(),
        ];

        return $this->success($summary);
    }
}
