<?php

namespace App\Http\Controllers\Api;

use App\Models\Employee;
use App\Models\LeaveBalance;
use App\Models\LeaveRequest;
use App\Models\LeaveType;
use App\Services\LeaveService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class LeaveController extends ApiController
{
    public function __construct(protected LeaveService $leaveService) {}

    public function types(Request $request): JsonResponse
    {
        $types = LeaveType::where('tenant_id', $request->user()->tenant_id)
            ->orWhereNull('tenant_id')
            ->where('is_active', true)
            ->get();

        return $this->success($types);
    }

    public function balances(Request $request): JsonResponse
    {
        $employee = Employee::where('user_id', $request->user()->id)->first();

        if (!$employee) {
            return $this->error('Employee record not found', 404);
        }

        $balances = LeaveBalance::where('employee_id', $employee->id)
            ->where('year', now()->year)
            ->with('leaveType')
            ->get()
            ->map(fn($b) => [
                'leave_type' => $b->leaveType->name,
                'entitled' => $b->entitled,
                'used' => $b->used,
                'carried_forward' => $b->carried_forward,
                'available' => $b->available,
            ]);

        return $this->success($balances);
    }

    public function index(Request $request): JsonResponse
    {
        $query = LeaveRequest::whereHas('employee', fn($q) => $q->where('tenant_id', $request->user()->tenant_id))
            ->with(['employee.user', 'leaveType', 'approvedBy']);

        // Filter by status
        if ($request->status) {
            $query->where('status', $request->status);
        }

        // Filter by date range
        if ($request->from_date) {
            $query->where('start_date', '>=', $request->from_date);
        }
        if ($request->to_date) {
            $query->where('end_date', '<=', $request->to_date);
        }

        // My requests only
        if ($request->my_requests) {
            $employee = Employee::where('user_id', $request->user()->id)->first();
            if ($employee) {
                $query->where('employee_id', $employee->id);
            }
        }

        // Pending approvals (for managers)
        if ($request->pending_approval) {
            $query->where('status', 'pending');
        }

        $requests = $query->orderBy('created_at', 'desc')->paginate($request->per_page ?? 20);

        return $this->paginated($requests);
    }

    public function store(Request $request): JsonResponse
    {
        $employee = Employee::where('user_id', $request->user()->id)->first();

        if (!$employee) {
            return $this->error('Employee record not found', 404);
        }

        $validated = $request->validate([
            'leave_type_id' => 'required|exists:leave_types,id',
            'start_date' => 'required|date|after_or_equal:today',
            'end_date' => 'required|date|after_or_equal:start_date',
            'start_half_day' => 'boolean',
            'end_half_day' => 'boolean',
            'reason' => 'nullable|string|max:500',
        ]);

        try {
            $leaveRequest = $this->leaveService->request($employee, $validated);
            return $this->success($leaveRequest->load(['leaveType', 'employee.user']), 'Leave request submitted', 201);
        } catch (\Exception $e) {
            return $this->error($e->getMessage(), 422);
        }
    }

    public function show(LeaveRequest $request): JsonResponse
    {
        return $this->success($request->load(['employee.user', 'leaveType', 'approvedBy']));
    }

    public function approve(Request $request, LeaveRequest $leaveRequest): JsonResponse
    {
        if ($leaveRequest->status !== 'pending') {
            return $this->error('Leave request is not pending', 422);
        }

        $this->leaveService->approve(
            $leaveRequest,
            $request->user()->id,
            $request->comments
        );

        return $this->success($leaveRequest->fresh()->load(['employee.user', 'leaveType']), 'Leave request approved');
    }

    public function reject(Request $request, LeaveRequest $leaveRequest): JsonResponse
    {
        if ($leaveRequest->status !== 'pending') {
            return $this->error('Leave request is not pending', 422);
        }

        $request->validate(['reason' => 'required|string|max:500']);

        $this->leaveService->reject($leaveRequest, $request->user()->id, $request->reason);

        return $this->success($leaveRequest->fresh(), 'Leave request rejected');
    }

    public function cancel(LeaveRequest $leaveRequest): JsonResponse
    {
        $employee = Employee::where('user_id', request()->user()->id)->first();

        if ($leaveRequest->employee_id !== $employee?->id) {
            return $this->error('Unauthorized', 403);
        }

        if (!in_array($leaveRequest->status, ['pending', 'approved'])) {
            return $this->error('Leave request cannot be cancelled', 422);
        }

        $this->leaveService->cancel($leaveRequest);

        return $this->success($leaveRequest->fresh(), 'Leave request cancelled');
    }

    public function calendar(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'month' => 'required|integer|between:1,12',
            'year' => 'required|integer|min:2020',
        ]);

        $leaves = LeaveRequest::whereHas('employee', fn($q) => $q->where('tenant_id', $request->user()->tenant_id))
            ->where('status', 'approved')
            ->whereYear('start_date', '<=', $validated['year'])
            ->whereYear('end_date', '>=', $validated['year'])
            ->whereMonth('start_date', '<=', $validated['month'])
            ->whereMonth('end_date', '>=', $validated['month'])
            ->with(['employee.user:id,first_name,last_name', 'leaveType:id,name,color'])
            ->get()
            ->map(fn($l) => [
                'id' => $l->id,
                'employee' => $l->employee->user->full_name,
                'type' => $l->leaveType->name,
                'color' => $l->leaveType->color,
                'start_date' => $l->start_date,
                'end_date' => $l->end_date,
            ]);

        return $this->success($leaves);
    }
}
