<?php

namespace App\Http\Controllers\Api;

use App\Models\Milestone;
use App\Models\Project;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class MilestoneController extends ApiController
{
    public function index(Request $request, Project $project): JsonResponse
    {
        $this->authorize('view', $project);

        $milestones = $project->milestones()
            ->with(['tasks' => fn($q) => $q->with('status')])
            ->orderBy('due_date')
            ->get();

        return $this->success($milestones);
    }

    public function store(Request $request, Project $project): JsonResponse
    {
        $this->authorize('update', $project);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'due_date' => 'nullable|date',
            'color' => 'nullable|string|max:7',
        ]);

        $milestone = $project->milestones()->create([
            'tenant_id' => $project->tenant_id,
            'name' => $validated['name'],
            'description' => $validated['description'] ?? null,
            'due_date' => $validated['due_date'] ?? null,
            'color' => $validated['color'] ?? '#2196F3',
            'created_by' => $request->user()->id,
        ]);

        return $this->success($milestone, 'Milestone created successfully', 201);
    }

    public function show(Request $request, Milestone $milestone): JsonResponse
    {
        $this->authorize('view', $milestone->project);
        
        return $this->success($milestone->load(['tasks.status', 'tasks.assignees.user']));
    }

    public function update(Request $request, Milestone $milestone): JsonResponse
    {
        $this->authorize('update', $milestone->project);

        $validated = $request->validate([
            'name' => 'sometimes|string|max:255',
            'description' => 'nullable|string',
            'due_date' => 'nullable|date',
            'color' => 'nullable|string|max:7',
        ]);

        $milestone->update($validated);

        return $this->success($milestone->fresh(), 'Milestone updated successfully');
    }

    public function destroy(Request $request, Milestone $milestone): JsonResponse
    {
        $this->authorize('update', $milestone->project);

        $milestone->tasks()->update(['milestone_id' => null]);
        $milestone->delete();

        return $this->success(null, 'Milestone deleted successfully');
    }

    public function complete(Request $request, Milestone $milestone): JsonResponse
    {
        $this->authorize('update', $milestone->project);

        $milestone->update([
            'status' => 'completed',
            'completed_at' => now(),
        ]);

        return $this->success($milestone->fresh(), 'Milestone completed');
    }
}
