<?php

namespace App\Http\Controllers\Api;

use App\Models\Pipeline;
use App\Models\PipelineStage;
use App\Models\Opportunity;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class PipelineController extends ApiController
{
    public function index(): JsonResponse
    {
        $pipelines = Pipeline::with('stages')->orderBy('position')->get();
        return $this->success($pipelines);
    }

    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'type' => 'nullable|string|max:50',
            'is_default' => 'boolean',
        ]);

        if ($validated['is_default'] ?? false) {
            Pipeline::where('is_default', true)->update(['is_default' => false]);
        }

        $pipeline = Pipeline::create($validated);

        // Create default stages
        $defaultStages = [
            ['name' => 'New', 'color' => '#6B7280', 'probability' => 10],
            ['name' => 'Contacted', 'color' => '#3B82F6', 'probability' => 25],
            ['name' => 'Qualified', 'color' => '#8B5CF6', 'probability' => 50],
            ['name' => 'Proposal', 'color' => '#F59E0B', 'probability' => 75],
            ['name' => 'Won', 'color' => '#10B981', 'probability' => 100, 'is_won' => true],
            ['name' => 'Lost', 'color' => '#EF4444', 'probability' => 0, 'is_lost' => true],
        ];

        foreach ($defaultStages as $index => $stage) {
            $pipeline->stages()->create(array_merge($stage, ['position' => $index]));
        }

        return $this->success($pipeline->load('stages'), 'Pipeline created successfully', 201);
    }

    public function show(Pipeline $pipeline): JsonResponse
    {
        return $this->success($pipeline->load(['stages', 'opportunities.contact']));
    }

    public function update(Request $request, Pipeline $pipeline): JsonResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'type' => 'nullable|string|max:50',
            'is_default' => 'boolean',
            'is_active' => 'boolean',
        ]);

        if ($validated['is_default'] ?? false) {
            Pipeline::where('id', '!=', $pipeline->id)->where('is_default', true)->update(['is_default' => false]);
        }

        $pipeline->update($validated);
        return $this->success($pipeline, 'Pipeline updated successfully');
    }

    public function destroy(Pipeline $pipeline): JsonResponse
    {
        if ($pipeline->opportunities()->exists()) {
            return $this->error('Cannot delete pipeline with opportunities', 422);
        }
        
        $pipeline->delete();
        return $this->success(null, 'Pipeline deleted successfully');
    }

    public function reorderStages(Request $request, Pipeline $pipeline): JsonResponse
    {
        $request->validate([
            'stages' => 'required|array',
            'stages.*.id' => 'required|exists:pipeline_stages,id',
            'stages.*.position' => 'required|integer',
        ]);

        foreach ($request->stages as $stageData) {
            PipelineStage::where('id', $stageData['id'])->update(['position' => $stageData['position']]);
        }

        return $this->success($pipeline->load('stages'), 'Stages reordered');
    }
}

class OpportunityController extends ApiController
{
    public function index(Request $request): JsonResponse
    {
        $query = Opportunity::with(['contact', 'company', 'owner', 'stage', 'pipeline'])
            ->when($request->pipeline_id, fn($q, $id) => $q->where('pipeline_id', $id))
            ->when($request->stage_id, fn($q, $id) => $q->where('stage_id', $id))
            ->when($request->status, fn($q, $s) => $q->where('status', $s))
            ->when($request->owner_id, fn($q, $id) => $q->where('owner_id', $id))
            ->when($request->search, fn($q, $s) => $q->where('title', 'like', "%{$s}%"))
            ->orderBy($request->sort_by ?? 'created_at', $request->sort_dir ?? 'desc');

        return $this->paginated($query->paginate($request->per_page ?? 25));
    }

    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'pipeline_id' => 'required|exists:pipelines,id',
            'stage_id' => 'required|exists:pipeline_stages,id',
            'contact_id' => 'nullable|exists:contacts,id',
            'company_id' => 'nullable|exists:contacts,id',
            'value' => 'nullable|numeric|min:0',
            'currency' => 'nullable|string|size:3',
            'expected_close_date' => 'nullable|date',
            'description' => 'nullable|string',
            'source' => 'nullable|string|max:100',
            'package' => 'nullable|in:standard,gold,premium',
        ]);

        $validated['owner_id'] = auth()->id();
        $validated['tenant_id'] = auth()->user()->tenant_id;

        $opportunity = Opportunity::create($validated);

        return $this->success($opportunity->load(['contact', 'stage', 'pipeline']), 'Opportunity created', 201);
    }

    public function show(Opportunity $opportunity): JsonResponse
    {
        return $this->success(
            $opportunity->load(['contact', 'company', 'owner', 'stage', 'pipeline', 'products', 'activities'])
        );
    }

    public function update(Request $request, Opportunity $opportunity): JsonResponse
    {
        $validated = $request->validate([
            'title' => 'sometimes|string|max:255',
            'stage_id' => 'sometimes|exists:pipeline_stages,id',
            'contact_id' => 'nullable|exists:contacts,id',
            'company_id' => 'nullable|exists:contacts,id',
            'value' => 'nullable|numeric|min:0',
            'expected_close_date' => 'nullable|date',
            'description' => 'nullable|string',
            'status' => 'sometimes|in:open,won,lost',
            'lost_reason' => 'nullable|string',
        ]);

        $oldStageId = $opportunity->stage_id;
        $opportunity->update($validated);

        // Log stage change
        if (isset($validated['stage_id']) && $oldStageId != $validated['stage_id']) {
            $opportunity->activities()->create([
                'tenant_id' => $opportunity->tenant_id,
                'user_id' => auth()->id(),
                'type' => 'stage_change',
                'title' => 'Stage changed',
                'description' => "Stage changed from {$oldStageId} to {$validated['stage_id']}",
            ]);
        }

        return $this->success($opportunity->fresh(['contact', 'stage']), 'Opportunity updated');
    }

    public function destroy(Opportunity $opportunity): JsonResponse
    {
        $opportunity->delete();
        return $this->success(null, 'Opportunity deleted');
    }

    public function moveStage(Request $request, Opportunity $opportunity): JsonResponse
    {
        $request->validate([
            'stage_id' => 'required|exists:pipeline_stages,id',
            'position' => 'nullable|integer',
        ]);

        $opportunity->update([
            'stage_id' => $request->stage_id,
            'position' => $request->position ?? $opportunity->position,
        ]);

        return $this->success($opportunity->fresh(['stage']), 'Opportunity moved');
    }

    public function kanban(Pipeline $pipeline): JsonResponse
    {
        $stages = $pipeline->stages()
            ->with(['opportunities' => fn($q) => $q->with(['contact', 'owner'])->orderBy('position')])
            ->orderBy('position')
            ->get();

        return $this->success([
            'pipeline' => $pipeline,
            'stages' => $stages,
        ]);
    }

    public function markWon(Opportunity $opportunity): JsonResponse
    {
        $wonStage = $opportunity->pipeline->stages()->where('is_won', true)->first();
        
        $opportunity->update([
            'status' => 'won',
            'stage_id' => $wonStage?->id ?? $opportunity->stage_id,
            'actual_close_date' => now(),
        ]);

        return $this->success($opportunity->fresh(), 'Opportunity marked as won');
    }

    public function markLost(Request $request, Opportunity $opportunity): JsonResponse
    {
        $request->validate(['lost_reason' => 'nullable|string']);

        $lostStage = $opportunity->pipeline->stages()->where('is_lost', true)->first();
        
        $opportunity->update([
            'status' => 'lost',
            'stage_id' => $lostStage?->id ?? $opportunity->stage_id,
            'actual_close_date' => now(),
            'lost_reason' => $request->lost_reason,
        ]);

        return $this->success($opportunity->fresh(), 'Opportunity marked as lost');
    }
}
