<?php

namespace App\Http\Controllers\Api;

use App\Models\Project;
use App\Models\Task;
use App\Models\TaskStatus;
use App\Models\TimeEntry;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class ProjectController extends ApiController
{
    public function index(Request $request): JsonResponse
    {
        $query = Project::with(['owner', 'workspace', 'contact'])
            ->withCount(['tasks', 'members'])
            ->when($request->status, fn($q, $s) => $q->where('status', $s))
            ->when($request->workspace_id, fn($q, $id) => $q->where('workspace_id', $id))
            ->when($request->owner_id, fn($q, $id) => $q->where('owner_id', $id))
            ->when($request->search, fn($q, $s) => $q->where('name', 'like', "%{$s}%"))
            ->when(!$request->include_archived, fn($q) => $q->where('status', '!=', 'archived'))
            ->orderBy($request->sort_by ?? 'created_at', $request->sort_dir ?? 'desc');

        return $this->paginated($query->paginate($request->per_page ?? 25));
    }

    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'nullable|string|max:20|unique:projects,code',
            'description' => 'nullable|string',
            'workspace_id' => 'nullable|exists:workspaces,id',
            'contact_id' => 'nullable|exists:contacts,id',
            'color' => 'nullable|string|max:7',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'budget' => 'nullable|numeric|min:0',
            'hourly_rate' => 'nullable|numeric|min:0',
            'estimated_hours' => 'nullable|integer|min:0',
            'is_billable' => 'boolean',
            'visibility' => 'nullable|in:private,team,public',
        ]);

        $validated['owner_id'] = auth()->id();
        $validated['tenant_id'] = auth()->user()->tenant_id;

        $project = Project::create($validated);

        // Create default statuses
        $defaultStatuses = [
            ['name' => 'To Do', 'color' => '#6B7280', 'is_default' => true],
            ['name' => 'In Progress', 'color' => '#3B82F6'],
            ['name' => 'Review', 'color' => '#F59E0B'],
            ['name' => 'Done', 'color' => '#10B981', 'is_completed' => true],
        ];

        foreach ($defaultStatuses as $index => $status) {
            $project->statuses()->create(array_merge($status, ['position' => $index]));
        }

        // Add creator as member
        $project->members()->attach(auth()->id(), ['role' => 'admin']);

        return $this->success($project->load(['owner', 'statuses', 'members']), 'Project created', 201);
    }

    public function show(Project $project): JsonResponse
    {
        return $this->success($project->load([
            'owner', 'workspace', 'contact', 'statuses', 'members.user', 
            'tasks' => fn($q) => $q->whereNull('parent_id')->with(['assignee', 'status']),
            'milestones'
        ]));
    }

    public function update(Request $request, Project $project): JsonResponse
    {
        $validated = $request->validate([
            'name' => 'sometimes|string|max:255',
            'description' => 'nullable|string',
            'status' => 'sometimes|in:planning,active,on_hold,completed,cancelled,archived',
            'color' => 'nullable|string|max:7',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date',
            'budget' => 'nullable|numeric|min:0',
            'hourly_rate' => 'nullable|numeric|min:0',
            'visibility' => 'nullable|in:private,team,public',
        ]);

        $project->update($validated);
        return $this->success($project->fresh(['owner', 'statuses']), 'Project updated');
    }

    public function destroy(Project $project): JsonResponse
    {
        $project->tasks()->delete();
        $project->statuses()->delete();
        $project->members()->detach();
        $project->delete();

        return $this->success(null, 'Project deleted');
    }

    public function members(Project $project): JsonResponse
    {
        return $this->success($project->members()->with('user')->get());
    }

    public function addMember(Request $request, Project $project): JsonResponse
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
            'role' => 'nullable|in:viewer,member,manager,admin',
        ]);

        $project->members()->syncWithoutDetaching([
            $request->user_id => ['role' => $request->role ?? 'member']
        ]);

        return $this->success($project->members()->with('user')->get(), 'Member added');
    }

    public function removeMember(Project $project, int $userId): JsonResponse
    {
        $project->members()->detach($userId);
        return $this->success(null, 'Member removed');
    }

    public function kanban(Project $project): JsonResponse
    {
        $statuses = $project->statuses()
            ->with(['tasks' => fn($q) => $q->whereNull('parent_id')->with(['assignee', 'subtasks'])->orderBy('position')])
            ->orderBy('position')
            ->get();

        return $this->success([
            'project' => $project,
            'statuses' => $statuses,
        ]);
    }

    public function gantt(Project $project): JsonResponse
    {
        $tasks = $project->tasks()
            ->whereNull('parent_id')
            ->with(['assignee', 'subtasks', 'dependencies'])
            ->whereNotNull('start_date')
            ->orderBy('start_date')
            ->get();

        $milestones = $project->milestones()->orderBy('due_date')->get();

        return $this->success([
            'project' => $project,
            'tasks' => $tasks,
            'milestones' => $milestones,
        ]);
    }

    public function stats(Project $project): JsonResponse
    {
        $totalTasks = $project->tasks()->count();
        $completedTasks = $project->tasks()->whereHas('status', fn($q) => $q->where('is_completed', true))->count();

        return $this->success([
            'total_tasks' => $totalTasks,
            'completed_tasks' => $completedTasks,
            'progress' => $totalTasks > 0 ? round(($completedTasks / $totalTasks) * 100) : 0,
            'total_hours_logged' => $project->timeEntries()->sum('hours'),
            'total_hours_estimated' => $project->tasks()->sum('estimated_hours'),
            'budget_spent' => $project->timeEntries()->where('is_billable', true)->sum('amount'),
            'members_count' => $project->members()->count(),
        ]);
    }
}

class TaskController extends ApiController
{
    public function index(Request $request): JsonResponse
    {
        $query = Task::with(['assignee', 'project', 'status', 'creator'])
            ->when($request->project_id, fn($q, $id) => $q->where('project_id', $id))
            ->when($request->status_id, fn($q, $id) => $q->where('status_id', $id))
            ->when($request->assignee_id, fn($q, $id) => $q->where('assignee_id', $id))
            ->when($request->priority, fn($q, $p) => $q->where('priority', $p))
            ->when($request->due_date, fn($q, $d) => $q->whereDate('due_date', $d))
            ->when($request->overdue, fn($q) => $q->whereDate('due_date', '<', now())->whereNull('completed_at'))
            ->when($request->search, fn($q, $s) => $q->where('title', 'like', "%{$s}%"))
            ->when(!$request->include_subtasks, fn($q) => $q->whereNull('parent_id'))
            ->orderBy($request->sort_by ?? 'position', $request->sort_dir ?? 'asc');

        return $this->paginated($query->paginate($request->per_page ?? 50));
    }

    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'project_id' => 'required|exists:projects,id',
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'status_id' => 'nullable|exists:task_statuses,id',
            'parent_id' => 'nullable|exists:tasks,id',
            'group_id' => 'nullable|exists:task_groups,id',
            'assignee_id' => 'nullable|exists:users,id',
            'priority' => 'nullable|in:low,medium,high,urgent',
            'type' => 'nullable|in:task,bug,feature,improvement,milestone,subtask',
            'start_date' => 'nullable|date',
            'due_date' => 'nullable|date',
            'estimated_hours' => 'nullable|numeric|min:0',
            'is_billable' => 'boolean',
        ]);

        $validated['creator_id'] = auth()->id();
        $validated['tenant_id'] = auth()->user()->tenant_id;

        // Set default status if not provided
        if (!isset($validated['status_id'])) {
            $defaultStatus = TaskStatus::where('project_id', $validated['project_id'])
                ->where('is_default', true)->first();
            $validated['status_id'] = $defaultStatus?->id;
        }

        $task = Task::create($validated);

        return $this->success($task->load(['assignee', 'status', 'project']), 'Task created', 201);
    }

    public function show(Task $task): JsonResponse
    {
        return $this->success($task->load([
            'assignee', 'creator', 'reviewer', 'status', 'project',
            'subtasks.assignee', 'subtasks.status',
            'comments.user', 'attachments', 'checklists.items',
            'dependencies', 'tags', 'timeEntries.user'
        ]));
    }

    public function update(Request $request, Task $task): JsonResponse
    {
        $validated = $request->validate([
            'title' => 'sometimes|string|max:255',
            'description' => 'nullable|string',
            'status_id' => 'sometimes|exists:task_statuses,id',
            'assignee_id' => 'nullable|exists:users,id',
            'priority' => 'sometimes|in:low,medium,high,urgent',
            'start_date' => 'nullable|date',
            'due_date' => 'nullable|date',
            'estimated_hours' => 'nullable|numeric|min:0',
            'progress' => 'nullable|integer|min:0|max:100',
        ]);

        $oldStatusId = $task->status_id;
        $task->update($validated);

        // Check if task is completed
        if (isset($validated['status_id']) && $oldStatusId != $validated['status_id']) {
            $newStatus = TaskStatus::find($validated['status_id']);
            if ($newStatus?->is_completed && !$task->completed_at) {
                $task->update(['completed_at' => now()]);
            } elseif (!$newStatus?->is_completed && $task->completed_at) {
                $task->update(['completed_at' => null]);
            }

            // Log activity
            $task->activities()->create([
                'user_id' => auth()->id(),
                'action' => 'status_changed',
                'field' => 'status_id',
                'old_value' => $oldStatusId,
                'new_value' => $validated['status_id'],
            ]);
        }

        return $this->success($task->fresh(['assignee', 'status']), 'Task updated');
    }

    public function destroy(Task $task): JsonResponse
    {
        $task->subtasks()->delete();
        $task->comments()->delete();
        $task->attachments()->delete();
        $task->delete();

        return $this->success(null, 'Task deleted');
    }

    public function move(Request $request, Task $task): JsonResponse
    {
        $request->validate([
            'status_id' => 'required|exists:task_statuses,id',
            'position' => 'nullable|integer',
        ]);

        $task->update([
            'status_id' => $request->status_id,
            'position' => $request->position ?? $task->position,
        ]);

        return $this->success($task->fresh(['status']), 'Task moved');
    }

    public function complete(Task $task): JsonResponse
    {
        $completedStatus = $task->project->statuses()->where('is_completed', true)->first();

        $task->update([
            'status_id' => $completedStatus?->id ?? $task->status_id,
            'completed_at' => now(),
            'progress' => 100,
        ]);

        return $this->success($task->fresh(), 'Task completed');
    }

    public function reopen(Task $task): JsonResponse
    {
        $defaultStatus = $task->project->statuses()->where('is_default', true)->first();

        $task->update([
            'status_id' => $defaultStatus?->id ?? $task->status_id,
            'completed_at' => null,
        ]);

        return $this->success($task->fresh(), 'Task reopened');
    }

    public function comment(Request $request, Task $task): JsonResponse
    {
        $request->validate([
            'content' => 'required|string',
            'parent_id' => 'nullable|exists:task_comments,id',
        ]);

        $comment = $task->comments()->create([
            'user_id' => auth()->id(),
            'content' => $request->content,
            'parent_id' => $request->parent_id,
        ]);

        return $this->success($comment->load('user'), 'Comment added', 201);
    }

    public function logTime(Request $request, Task $task): JsonResponse
    {
        $request->validate([
            'hours' => 'required|numeric|min:0.01',
            'date' => 'required|date',
            'description' => 'nullable|string',
            'is_billable' => 'boolean',
        ]);

        $timeEntry = $task->timeEntries()->create([
            'tenant_id' => $task->tenant_id,
            'user_id' => auth()->id(),
            'project_id' => $task->project_id,
            'hours' => $request->hours,
            'date' => $request->date,
            'description' => $request->description,
            'is_billable' => $request->is_billable ?? $task->is_billable,
            'hourly_rate' => $task->project->hourly_rate ?? auth()->user()->hourly_rate,
        ]);

        $timeEntry->amount = $timeEntry->hours * ($timeEntry->hourly_rate ?? 0);
        $timeEntry->save();

        // Update task actual hours
        $task->actual_hours = $task->timeEntries()->sum('hours');
        $task->save();

        return $this->success($timeEntry->load('user'), 'Time logged', 201);
    }

    public function myTasks(Request $request): JsonResponse
    {
        $tasks = Task::with(['project', 'status'])
            ->where('assignee_id', auth()->id())
            ->when($request->status, fn($q, $s) => 
                $s === 'completed' 
                    ? $q->whereNotNull('completed_at')
                    : $q->whereNull('completed_at')
            )
            ->when($request->due, fn($q, $d) => 
                match($d) {
                    'today' => $q->whereDate('due_date', today()),
                    'week' => $q->whereBetween('due_date', [now(), now()->addWeek()]),
                    'overdue' => $q->whereDate('due_date', '<', today())->whereNull('completed_at'),
                    default => $q
                }
            )
            ->orderBy('due_date')
            ->paginate($request->per_page ?? 25);

        return $this->paginated($tasks);
    }
}
