<?php

namespace App\Http\Controllers\Api;

use App\Models\RecurringInvoice;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class RecurringInvoiceController extends ApiController
{
    public function index(Request $request): JsonResponse
    {
        $query = RecurringInvoice::where('tenant_id', $request->user()->tenant_id)
            ->with(['contact']);

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $recurring = $query->orderBy('next_invoice_date')
            ->paginate($request->get('per_page', 15));

        return $this->paginated($recurring);
    }

    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'contact_id' => 'required|exists:contacts,id',
            'name' => 'required|string|max:255',
            'frequency' => 'required|in:daily,weekly,monthly,quarterly,yearly',
            'frequency_interval' => 'nullable|integer|min:1|max:12',
            'start_date' => 'required|date|after_or_equal:today',
            'end_date' => 'nullable|date|after:start_date',
            'payment_terms' => 'nullable|integer|min:0',
            'currency' => 'nullable|string|size:3',
            'notes' => 'nullable|string',
            'terms' => 'nullable|string',
            'send_automatically' => 'nullable|boolean',
            'max_invoices' => 'nullable|integer|min:1',
            'items' => 'required|array|min:1',
            'items.*.name' => 'required|string|max:255',
            'items.*.quantity' => 'required|numeric|min:0.0001',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.tax_rate_id' => 'nullable|exists:tax_rates,id',
        ]);

        $recurring = RecurringInvoice::create([
            'tenant_id' => $request->user()->tenant_id,
            'contact_id' => $validated['contact_id'],
            'name' => $validated['name'],
            'frequency' => $validated['frequency'],
            'frequency_interval' => $validated['frequency_interval'] ?? 1,
            'start_date' => $validated['start_date'],
            'end_date' => $validated['end_date'] ?? null,
            'next_invoice_date' => $validated['start_date'],
            'payment_terms' => $validated['payment_terms'] ?? 14,
            'currency' => $validated['currency'] ?? 'BHD',
            'notes' => $validated['notes'] ?? null,
            'terms' => $validated['terms'] ?? null,
            'send_automatically' => $validated['send_automatically'] ?? true,
            'max_invoices' => $validated['max_invoices'] ?? null,
            'created_by' => $request->user()->id,
        ]);

        foreach ($validated['items'] as $index => $item) {
            $recurring->items()->create(array_merge($item, [
                'tenant_id' => $recurring->tenant_id,
                'position' => $index + 1,
            ]));
        }

        $recurring->calculateTotals();
        $recurring->save();

        return $this->success($recurring->load(['contact', 'items']), 'Recurring invoice created successfully', 201);
    }

    public function show(Request $request, RecurringInvoice $recurringInvoice): JsonResponse
    {
        $this->authorize('view', $recurringInvoice);
        return $this->success($recurringInvoice->load(['contact', 'items.taxRate', 'invoices']));
    }

    public function update(Request $request, RecurringInvoice $recurringInvoice): JsonResponse
    {
        $this->authorize('update', $recurringInvoice);

        $validated = $request->validate([
            'contact_id' => 'sometimes|exists:contacts,id',
            'name' => 'sometimes|string|max:255',
            'frequency' => 'sometimes|in:daily,weekly,monthly,quarterly,yearly',
            'frequency_interval' => 'nullable|integer|min:1|max:12',
            'end_date' => 'nullable|date',
            'payment_terms' => 'nullable|integer|min:0',
            'notes' => 'nullable|string',
            'terms' => 'nullable|string',
            'send_automatically' => 'nullable|boolean',
            'items' => 'sometimes|array|min:1',
        ]);

        $recurringInvoice->update($validated);

        if (isset($validated['items'])) {
            $recurringInvoice->items()->delete();
            foreach ($validated['items'] as $index => $item) {
                $recurringInvoice->items()->create(array_merge($item, [
                    'tenant_id' => $recurringInvoice->tenant_id,
                    'position' => $index + 1,
                ]));
            }
            $recurringInvoice->calculateTotals();
            $recurringInvoice->save();
        }

        return $this->success($recurringInvoice->load(['contact', 'items']), 'Recurring invoice updated successfully');
    }

    public function destroy(Request $request, RecurringInvoice $recurringInvoice): JsonResponse
    {
        $this->authorize('delete', $recurringInvoice);
        $recurringInvoice->delete();
        return $this->success(null, 'Recurring invoice deleted successfully');
    }

    public function pause(Request $request, RecurringInvoice $recurringInvoice): JsonResponse
    {
        $this->authorize('update', $recurringInvoice);
        $recurringInvoice->update(['status' => RecurringInvoice::STATUS_PAUSED]);
        return $this->success($recurringInvoice->fresh(), 'Recurring invoice paused');
    }

    public function resume(Request $request, RecurringInvoice $recurringInvoice): JsonResponse
    {
        $this->authorize('update', $recurringInvoice);
        $recurringInvoice->update(['status' => RecurringInvoice::STATUS_ACTIVE]);
        return $this->success($recurringInvoice->fresh(), 'Recurring invoice resumed');
    }

    public function generateNow(Request $request, RecurringInvoice $recurringInvoice): JsonResponse
    {
        $this->authorize('update', $recurringInvoice);
        $invoice = $recurringInvoice->generateInvoice();
        return $this->success($invoice->load(['contact', 'items']), 'Invoice generated successfully', 201);
    }
}
