<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Segment;
use Illuminate\Http\Request;

class SegmentController extends Controller
{
    public function index(Request $request)
    {
        $segments = Segment::where('tenant_id', $request->user()->tenant_id)
            ->withCount('contacts')
            ->orderBy('name')
            ->get();

        return response()->json(['data' => $segments]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string|max:500',
            'is_dynamic' => 'boolean',
            'conditions' => 'array',
        ]);

        $segment = Segment::create([
            'tenant_id' => $request->user()->tenant_id,
            ...$validated,
        ]);

        if ($segment->is_dynamic) {
            $segment->refreshContacts();
        }

        return response()->json(['data' => $segment->loadCount('contacts')], 201);
    }

    public function show(Segment $segment)
    {
        $this->authorize('view', $segment);
        return response()->json(['data' => $segment->loadCount('contacts')]);
    }

    public function update(Request $request, Segment $segment)
    {
        $this->authorize('update', $segment);

        $segment->update($request->validate([
            'name' => 'sometimes|string|max:255',
            'description' => 'nullable|string|max:500',
            'is_dynamic' => 'boolean',
            'conditions' => 'array',
        ]));

        if ($segment->is_dynamic) {
            $segment->refreshContacts();
        }

        return response()->json(['data' => $segment->loadCount('contacts')]);
    }

    public function destroy(Segment $segment)
    {
        $this->authorize('delete', $segment);
        $segment->delete();
        return response()->noContent();
    }

    public function addContacts(Request $request, Segment $segment)
    {
        $this->authorize('update', $segment);

        $validated = $request->validate([
            'contact_ids' => 'required|array',
            'contact_ids.*' => 'exists:contacts,id',
        ]);

        $segment->contacts()->syncWithoutDetaching($validated['contact_ids']);

        return response()->json(['data' => $segment->loadCount('contacts')]);
    }

    public function removeContacts(Request $request, Segment $segment)
    {
        $this->authorize('update', $segment);

        $validated = $request->validate([
            'contact_ids' => 'required|array',
            'contact_ids.*' => 'exists:contacts,id',
        ]);

        $segment->contacts()->detach($validated['contact_ids']);

        return response()->json(['data' => $segment->loadCount('contacts')]);
    }

    public function refresh(Segment $segment)
    {
        $this->authorize('update', $segment);

        if ($segment->is_dynamic) {
            $segment->refreshContacts();
        }

        return response()->json(['data' => $segment->loadCount('contacts')]);
    }

    public function duplicate(Segment $segment)
    {
        $this->authorize('update', $segment);

        $newSegment = $segment->replicate();
        $newSegment->name = $segment->name . ' (Copy)';
        $newSegment->save();

        if (!$newSegment->is_dynamic) {
            $newSegment->contacts()->sync($segment->contacts->pluck('id'));
        }

        return response()->json(['data' => $newSegment->loadCount('contacts')], 201);
    }
}
