<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\ApiKey;
use App\Models\User;
use App\Models\Webhook;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class SettingsController extends Controller
{
    public function getCompany(Request $request)
    {
        return response()->json([
            'data' => $request->user()->tenant,
        ]);
    }

    public function updateCompany(Request $request)
    {
        $tenant = $request->user()->tenant;
        
        $tenant->update($request->validate([
            'name' => 'sometimes|string|max:255',
            'legal_name' => 'nullable|string|max:255',
            'cr_number' => 'nullable|string|max:50',
            'tax_number' => 'nullable|string|max:50',
            'email' => 'nullable|email|max:255',
            'phone' => 'nullable|string|max:50',
            'website' => 'nullable|url|max:255',
            'address' => 'nullable|string|max:500',
            'city' => 'nullable|string|max:100',
            'country' => 'nullable|string|size:2',
        ]));

        return response()->json(['data' => $tenant]);
    }

    public function getTeam(Request $request)
    {
        $users = User::where('tenant_id', $request->user()->tenant_id)
            ->orderBy('first_name')
            ->get();

        return response()->json(['data' => $users]);
    }

    public function getApiKeys(Request $request)
    {
        $keys = ApiKey::where('tenant_id', $request->user()->tenant_id)
            ->orderByDesc('created_at')
            ->get()
            ->map(fn($key) => [
                'id' => $key->id,
                'name' => $key->name,
                'key_preview' => substr($key->key, 0, 8) . '...' . substr($key->key, -4),
                'last_used_at' => $key->last_used_at,
                'created_at' => $key->created_at,
            ]);

        return response()->json(['data' => $keys]);
    }

    public function createApiKey(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:100',
        ]);

        $key = Str::random(40);
        
        $apiKey = ApiKey::create([
            'tenant_id' => $request->user()->tenant_id,
            'user_id' => $request->user()->id,
            'name' => $validated['name'],
            'key' => hash('sha256', $key),
        ]);

        return response()->json([
            'key' => $key,
            'data' => [
                'id' => $apiKey->id,
                'name' => $apiKey->name,
            ],
        ], 201);
    }

    public function deleteApiKey(ApiKey $apiKey)
    {
        $this->authorize('delete', $apiKey);
        
        $apiKey->delete();
        
        return response()->noContent();
    }

    public function getWebhooks(Request $request)
    {
        $webhooks = Webhook::where('tenant_id', $request->user()->tenant_id)
            ->orderByDesc('created_at')
            ->get();

        return response()->json(['data' => $webhooks]);
    }

    public function createWebhook(Request $request)
    {
        $validated = $request->validate([
            'url' => 'required|url',
            'events' => 'required|array|min:1',
            'events.*' => 'string',
        ]);

        $webhook = Webhook::create([
            'tenant_id' => $request->user()->tenant_id,
            'url' => $validated['url'],
            'events' => $validated['events'],
            'secret' => Str::random(32),
            'active' => true,
        ]);

        return response()->json(['data' => $webhook], 201);
    }

    public function updateWebhook(Request $request, Webhook $webhook)
    {
        $this->authorize('update', $webhook);
        
        $webhook->update($request->validate([
            'url' => 'sometimes|url',
            'events' => 'sometimes|array|min:1',
            'active' => 'sometimes|boolean',
        ]));

        return response()->json(['data' => $webhook]);
    }

    public function deleteWebhook(Webhook $webhook)
    {
        $this->authorize('delete', $webhook);
        
        $webhook->delete();
        
        return response()->noContent();
    }
}
