<?php

namespace App\Http\Controllers\Api;

use App\Models\SmsMessage;
use App\Models\Contact;
use App\Services\TwilioService;
use Illuminate\Http\Request;

class SmsController extends ApiController
{
    protected TwilioService $twilioService;

    public function __construct(TwilioService $twilioService)
    {
        $this->twilioService = $twilioService;
    }

    public function index(Request $request)
    {
        $query = SmsMessage::where('tenant_id', $request->user()->tenant_id)
            ->with('contact');

        if ($request->direction) {
            $query->where('direction', $request->direction);
        }

        if ($request->status) {
            $query->where('status', $request->status);
        }

        if ($request->contact_id) {
            $query->where('contact_id', $request->contact_id);
        }

        if ($request->search) {
            $query->where('message', 'like', "%{$request->search}%");
        }

        $messages = $query->latest()->paginate($request->per_page ?? 15);

        return $this->paginated($messages);
    }

    public function send(Request $request)
    {
        $validated = $request->validate([
            'to' => 'required|string',
            'message' => 'required|string|max:1600',
            'contact_id' => 'nullable|exists:contacts,id',
        ]);

        $tenantId = $request->user()->tenant_id;
        
        // Calculate segments
        $segments = SmsMessage::calculateSegments($validated['message']);

        // Create record
        $sms = SmsMessage::create([
            'tenant_id' => $tenantId,
            'contact_id' => $validated['contact_id'],
            'direction' => 'outbound',
            'from_number' => config('services.twilio.from_number'),
            'to_number' => $validated['to'],
            'message' => $validated['message'],
            'segments' => $segments,
            'status' => 'queued',
        ]);

        try {
            $result = $this->twilioService->sendSms($validated['to'], $validated['message']);
            
            $sms->update([
                'status' => 'sent',
                'sent_at' => now(),
                'provider_id' => $result->sid,
                'provider_response' => ['sid' => $result->sid, 'status' => $result->status],
            ]);
        } catch (\Exception $e) {
            $sms->update([
                'status' => 'failed',
                'failed_at' => now(),
                'error_message' => $e->getMessage(),
            ]);

            return $this->error('Failed to send SMS: ' . $e->getMessage(), 500);
        }

        return $this->success($sms, 'SMS sent successfully', 201);
    }

    public function sendBulk(Request $request)
    {
        $validated = $request->validate([
            'recipients' => 'required|array|min:1',
            'recipients.*.phone' => 'required|string',
            'recipients.*.contact_id' => 'nullable|exists:contacts,id',
            'message' => 'required|string|max:1600',
        ]);

        $tenantId = $request->user()->tenant_id;
        $segments = SmsMessage::calculateSegments($validated['message']);
        $results = ['sent' => 0, 'failed' => 0];

        foreach ($validated['recipients'] as $recipient) {
            $sms = SmsMessage::create([
                'tenant_id' => $tenantId,
                'contact_id' => $recipient['contact_id'] ?? null,
                'direction' => 'outbound',
                'from_number' => config('services.twilio.from_number'),
                'to_number' => $recipient['phone'],
                'message' => $validated['message'],
                'segments' => $segments,
                'status' => 'queued',
            ]);

            try {
                $result = $this->twilioService->sendSms($recipient['phone'], $validated['message']);
                
                $sms->update([
                    'status' => 'sent',
                    'sent_at' => now(),
                    'provider_id' => $result->sid,
                ]);
                
                $results['sent']++;
            } catch (\Exception $e) {
                $sms->update([
                    'status' => 'failed',
                    'failed_at' => now(),
                    'error_message' => $e->getMessage(),
                ]);
                
                $results['failed']++;
            }
        }

        return $this->success($results, 'Bulk SMS completed');
    }

    public function conversation(Contact $contact, Request $request)
    {
        $messages = SmsMessage::where('tenant_id', $request->user()->tenant_id)
            ->where('contact_id', $contact->id)
            ->orderBy('created_at')
            ->get();

        return $this->success([
            'contact' => $contact,
            'messages' => $messages,
        ]);
    }

    public function reply(Contact $contact, Request $request)
    {
        $validated = $request->validate([
            'message' => 'required|string|max:1600',
        ]);

        $sms = SmsMessage::create([
            'tenant_id' => $request->user()->tenant_id,
            'contact_id' => $contact->id,
            'direction' => 'outbound',
            'from_number' => config('services.twilio.from_number'),
            'to_number' => $contact->phone,
            'message' => $validated['message'],
            'segments' => SmsMessage::calculateSegments($validated['message']),
            'status' => 'queued',
        ]);

        try {
            $result = $this->twilioService->sendSms($contact->phone, $validated['message']);
            
            $sms->update([
                'status' => 'sent',
                'sent_at' => now(),
                'provider_id' => $result->sid,
            ]);
        } catch (\Exception $e) {
            $sms->update([
                'status' => 'failed',
                'error_message' => $e->getMessage(),
            ]);

            return $this->error('Failed to send SMS', 500);
        }

        return $this->success($sms, 'Reply sent', 201);
    }

    /**
     * Twilio webhook for inbound SMS
     */
    public function inbound(Request $request)
    {
        $from = $request->input('From');
        $to = $request->input('To');
        $body = $request->input('Body');
        $messageSid = $request->input('MessageSid');

        // Find contact by phone number
        $contact = Contact::where('phone', $from)
            ->orWhere('phone', '+' . ltrim($from, '+'))
            ->first();

        $tenantId = $contact?->tenant_id ?? config('app.default_tenant_id');

        SmsMessage::create([
            'tenant_id' => $tenantId,
            'contact_id' => $contact?->id,
            'direction' => 'inbound',
            'from_number' => $from,
            'to_number' => $to,
            'message' => $body,
            'segments' => SmsMessage::calculateSegments($body),
            'status' => 'received',
            'provider_id' => $messageSid,
        ]);

        return response('<?xml version="1.0" encoding="UTF-8"?><Response></Response>')
            ->header('Content-Type', 'text/xml');
    }

    public function statistics(Request $request)
    {
        $tenantId = $request->user()->tenant_id;
        $startDate = $request->start_date ?? now()->startOfMonth();
        $endDate = $request->end_date ?? now();

        return $this->success([
            'total_sent' => SmsMessage::where('tenant_id', $tenantId)
                ->where('direction', 'outbound')
                ->whereBetween('created_at', [$startDate, $endDate])
                ->count(),
            'total_received' => SmsMessage::where('tenant_id', $tenantId)
                ->where('direction', 'inbound')
                ->whereBetween('created_at', [$startDate, $endDate])
                ->count(),
            'total_failed' => SmsMessage::where('tenant_id', $tenantId)
                ->where('status', 'failed')
                ->whereBetween('created_at', [$startDate, $endDate])
                ->count(),
            'total_segments' => SmsMessage::where('tenant_id', $tenantId)
                ->where('direction', 'outbound')
                ->whereBetween('created_at', [$startDate, $endDate])
                ->sum('segments'),
        ]);
    }
}
