<?php

namespace App\Http\Controllers\Api;

use App\Models\Task;
use App\Models\TaskComment;
use App\Services\TaskService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class TaskController extends ApiController
{
    public function __construct(protected TaskService $taskService) {}

    public function index(Request $request): JsonResponse
    {
        $query = Task::where('tenant_id', $request->user()->tenant_id)
            ->with(['status', 'assignees.user', 'project:id,name,color']);

        if ($request->filled('project_id')) {
            $query->where('project_id', $request->project_id);
        }

        if ($request->filled('status_id')) {
            $query->where('status_id', $request->status_id);
        }

        if ($request->filled('assignee_id')) {
            $query->whereHas('assignees', fn($q) => $q->where('user_id', $request->assignee_id));
        }

        if ($request->filled('priority')) {
            $query->where('priority', $request->priority);
        }

        if ($request->filled('due_date')) {
            $query->whereDate('due_date', $request->due_date);
        }

        if ($request->filled('overdue')) {
            $query->where('due_date', '<', now())
                  ->whereHas('status', fn($q) => $q->where('is_completed', false));
        }

        if ($request->filled('search')) {
            $query->where('title', 'like', "%{$request->search}%");
        }

        if ($request->filled('my_tasks')) {
            $query->whereHas('assignees', fn($q) => $q->where('user_id', $request->user()->id));
        }

        $tasks = $query->orderBy($request->get('sort_by', 'position'))
            ->orderBy('created_at', 'desc')
            ->paginate($request->get('per_page', 50));

        return $this->paginated($tasks);
    }

    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'project_id' => 'required|exists:projects,id',
            'parent_id' => 'nullable|exists:tasks,id',
            'status_id' => 'nullable|exists:task_statuses,id',
            'group_id' => 'nullable|exists:task_groups,id',
            'milestone_id' => 'nullable|exists:milestones,id',
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'type' => 'nullable|in:task,bug,feature,story,epic',
            'priority' => 'nullable|in:urgent,high,medium,low',
            'start_date' => 'nullable|date',
            'due_date' => 'nullable|date|after_or_equal:start_date',
            'estimated_hours' => 'nullable|numeric|min:0',
            'assignees' => 'nullable|array',
            'assignees.*' => 'exists:users,id',
            'tags' => 'nullable|array',
        ]);

        $task = $this->taskService->create(
            $request->user()->tenant_id,
            $validated,
            $request->user()->id
        );

        return $this->success($task->load(['status', 'assignees.user']), 'Task created', 201);
    }

    public function show(Task $task): JsonResponse
    {
        $this->authorize('view', $task);

        return $this->success($task->load([
            'project', 'status', 'assignees.user', 'parent', 'subtasks.status',
            'checklists.items', 'tags', 'attachments', 'createdBy'
        ]));
    }

    public function update(Request $request, Task $task): JsonResponse
    {
        $this->authorize('update', $task);

        $validated = $request->validate([
            'parent_id' => 'nullable|exists:tasks,id',
            'status_id' => 'nullable|exists:task_statuses,id',
            'group_id' => 'nullable|exists:task_groups,id',
            'milestone_id' => 'nullable|exists:milestones,id',
            'title' => 'sometimes|string|max:255',
            'description' => 'nullable|string',
            'type' => 'nullable|in:task,bug,feature,story,epic',
            'priority' => 'nullable|in:urgent,high,medium,low',
            'start_date' => 'nullable|date',
            'due_date' => 'nullable|date',
            'estimated_hours' => 'nullable|numeric|min:0',
            'assignees' => 'nullable|array',
            'tags' => 'nullable|array',
            'position' => 'nullable|integer',
        ]);

        $task = $this->taskService->update($task, $validated, $request->user()->id);

        return $this->success($task->load(['status', 'assignees.user']), 'Task updated');
    }

    public function destroy(Task $task): JsonResponse
    {
        $this->authorize('delete', $task);

        $task->delete();

        return $this->success(null, 'Task deleted');
    }

    public function complete(Request $request, Task $task): JsonResponse
    {
        $this->authorize('update', $task);

        $this->taskService->complete($task, $request->user()->id);

        return $this->success($task->fresh(['status']), 'Task completed');
    }

    public function reopen(Task $task): JsonResponse
    {
        $this->authorize('update', $task);

        $this->taskService->reopen($task);

        return $this->success($task->fresh(['status']), 'Task reopened');
    }

    public function comments(Task $task): JsonResponse
    {
        $this->authorize('view', $task);

        $comments = $task->comments()
            ->with(['user', 'replies.user'])
            ->whereNull('parent_id')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return $this->paginated($comments);
    }

    public function addComment(Request $request, Task $task): JsonResponse
    {
        $this->authorize('update', $task);

        $validated = $request->validate([
            'content' => 'required|string',
            'parent_id' => 'nullable|exists:task_comments,id',
            'mentions' => 'nullable|array',
        ]);

        $comment = TaskComment::create([
            'tenant_id' => $task->tenant_id,
            'task_id' => $task->id,
            'user_id' => $request->user()->id,
            'parent_id' => $validated['parent_id'] ?? null,
            'content' => $validated['content'],
            'mentions' => $validated['mentions'] ?? null,
        ]);

        return $this->success($comment->load('user'), 'Comment added', 201);
    }

    public function activity(Task $task): JsonResponse
    {
        $this->authorize('view', $task);

        $activities = $task->activities()
            ->with('user')
            ->orderBy('created_at', 'desc')
            ->paginate(50);

        return $this->paginated($activities);
    }
}
