<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Resources\TimeEntryResource;
use App\Models\TimeEntry;
use App\Services\TimeEntryService;
use Illuminate\Http\Request;

class TimeEntryController extends Controller
{
    public function __construct(protected TimeEntryService $service) {}

    public function index(Request $request)
    {
        $entries = TimeEntry::where('tenant_id', $request->user()->tenant_id)
            ->when($request->user_id, fn($q, $v) => $q->where('user_id', $v))
            ->when($request->task_id, fn($q, $v) => $q->where('task_id', $v))
            ->when($request->start_date, fn($q, $v) => $q->whereDate('start_time', '>=', $v))
            ->when($request->end_date, fn($q, $v) => $q->whereDate('start_time', '<=', $v))
            ->with(['user', 'task.project'])
            ->orderByDesc('start_time')
            ->paginate($request->per_page ?? 20);

        return TimeEntryResource::collection($entries);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'task_id' => 'required|exists:tasks,id',
            'start_time' => 'required|date',
            'end_time' => 'required|date|after:start_time',
            'description' => 'nullable|string|max:500',
            'is_billable' => 'boolean',
        ]);

        $entry = $this->service->manual(
            $request->user()->tenant_id,
            $request->user()->id,
            $validated
        );

        return new TimeEntryResource($entry);
    }

    public function start(Request $request)
    {
        $validated = $request->validate([
            'task_id' => 'required|exists:tasks,id',
            'description' => 'nullable|string|max:500',
        ]);

        $entry = $this->service->start(
            $request->user()->tenant_id,
            $request->user()->id,
            $validated['task_id'],
            $validated['description'] ?? null
        );

        return new TimeEntryResource($entry);
    }

    public function stop(TimeEntry $timeEntry)
    {
        $this->authorize('update', $timeEntry);
        
        $entry = $this->service->stop($timeEntry);
        
        return new TimeEntryResource($entry);
    }

    public function running(Request $request)
    {
        $entry = $this->service->getRunning(
            $request->user()->tenant_id,
            $request->user()->id
        );

        return response()->json([
            'data' => $entry ? new TimeEntryResource($entry) : null,
        ]);
    }

    public function summary(Request $request)
    {
        $startDate = $request->input('start_date', now()->startOfWeek()->toDateString());
        $endDate = $request->input('end_date', now()->toDateString());
        
        $summary = $this->service->getUserSummary(
            $request->user()->tenant_id,
            $request->user()->id,
            $startDate,
            $endDate
        );

        return response()->json(['data' => $summary]);
    }

    public function update(Request $request, TimeEntry $timeEntry)
    {
        $this->authorize('update', $timeEntry);
        
        $timeEntry->update($request->validate([
            'description' => 'nullable|string|max:500',
            'is_billable' => 'boolean',
        ]));

        return new TimeEntryResource($timeEntry);
    }

    public function destroy(TimeEntry $timeEntry)
    {
        $this->authorize('delete', $timeEntry);
        
        $timeEntry->delete();
        
        return response()->noContent();
    }
}
