<?php

namespace App\Http\Controllers\Api;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class UserController extends ApiController
{
    public function index(Request $request)
    {
        $query = User::where('tenant_id', $request->user()->tenant_id);

        if ($request->search) {
            $query->where(function ($q) use ($request) {
                $q->where('name', 'like', "%{$request->search}%")
                  ->orWhere('email', 'like', "%{$request->search}%");
            });
        }

        if ($request->role) {
            $query->where('role', $request->role);
        }

        if ($request->has('is_active')) {
            $query->where('is_active', $request->boolean('is_active'));
        }

        $users = $query->with('teams')
            ->orderBy($request->sort_by ?? 'name', $request->sort_order ?? 'asc')
            ->paginate($request->per_page ?? 15);

        return $this->paginated($users);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users',
            'password' => 'required|min:8',
            'phone' => 'nullable|string|max:20',
            'role' => 'required|in:admin,manager,user',
            'team_ids' => 'nullable|array',
            'team_ids.*' => 'exists:teams,id',
        ]);

        $user = User::create([
            'tenant_id' => $request->user()->tenant_id,
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'phone' => $validated['phone'] ?? null,
            'role' => $validated['role'],
        ]);

        if (!empty($validated['team_ids'])) {
            $user->teams()->attach($validated['team_ids']);
        }

        return $this->success($user->load('teams'), 'User created successfully', 201);
    }

    public function show(User $user)
    {
        $this->authorize('view', $user);
        return $this->success($user->load('teams'));
    }

    public function update(Request $request, User $user)
    {
        $this->authorize('update', $user);

        $validated = $request->validate([
            'name' => 'sometimes|string|max:255',
            'email' => 'sometimes|email|unique:users,email,' . $user->id,
            'phone' => 'nullable|string|max:20',
            'role' => 'sometimes|in:admin,manager,user',
            'team_ids' => 'nullable|array',
        ]);

        $user->update($validated);

        if (isset($validated['team_ids'])) {
            $user->teams()->sync($validated['team_ids']);
        }

        return $this->success($user->load('teams'), 'User updated successfully');
    }

    public function destroy(User $user)
    {
        $this->authorize('delete', $user);

        if ($user->id === auth()->id()) {
            return $this->error('Cannot delete your own account', 400);
        }

        $user->delete();

        return $this->success(null, 'User deleted successfully');
    }

    public function deactivate(User $user)
    {
        $this->authorize('update', $user);

        if ($user->id === auth()->id()) {
            return $this->error('Cannot deactivate your own account', 400);
        }

        $user->update(['is_active' => false]);
        $user->tokens()->delete();

        return $this->success($user, 'User deactivated successfully');
    }

    public function activate(User $user)
    {
        $this->authorize('update', $user);
        $user->update(['is_active' => true]);
        return $this->success($user, 'User activated successfully');
    }

    public function updateRoles(Request $request, User $user)
    {
        $this->authorize('update', $user);

        $validated = $request->validate([
            'role' => 'required|in:admin,manager,user',
        ]);

        $user->update(['role' => $validated['role']]);

        return $this->success($user, 'User role updated successfully');
    }

    public function activity(User $user)
    {
        $this->authorize('view', $user);

        $activities = Activity::where('user_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return $this->paginated($activities);
    }
}
