<?php

namespace App\Http\Controllers\Api;

use App\Models\Workflow;
use App\Models\WorkflowEnrollment;
use App\Services\WorkflowService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;

class WorkflowController extends ApiController
{
    public function __construct(protected WorkflowService $workflowService) {}

    public function index(Request $request): JsonResponse
    {
        $workflows = Workflow::where('tenant_id', $request->user()->tenant_id)
            ->withCount(['enrollments', 'enrollments as active_enrollments_count' => fn($q) => $q->where('status', 'active')])
            ->orderBy('created_at', 'desc')
            ->paginate($request->per_page ?? 20);

        return $this->paginated($workflows);
    }

    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'trigger' => 'required|array',
            'trigger.type' => 'required|string',
            'trigger.conditions' => 'nullable|array',
            'steps' => 'required|array|min:1',
            'steps.*.type' => 'required|string',
            'steps.*.config' => 'required|array',
            'steps.*.delay' => 'nullable|array',
        ]);

        $workflow = DB::transaction(function () use ($validated, $request) {
            $workflow = Workflow::create([
                'tenant_id' => $request->user()->tenant_id,
                'name' => $validated['name'],
                'description' => $validated['description'] ?? null,
                'trigger' => $validated['trigger'],
                'is_active' => false,
                'created_by' => $request->user()->id,
            ]);

            foreach ($validated['steps'] as $index => $step) {
                $workflow->steps()->create([
                    'tenant_id' => $request->user()->tenant_id,
                    'type' => $step['type'],
                    'config' => $step['config'],
                    'delay' => $step['delay'] ?? null,
                    'position' => $index + 1,
                ]);
            }

            return $workflow;
        });

        return $this->success($workflow->load('steps'), 'Workflow created', 201);
    }

    public function show(Workflow $workflow): JsonResponse
    {
        $this->authorize('view', $workflow);
        return $this->success($workflow->load('steps'));
    }

    public function update(Request $request, Workflow $workflow): JsonResponse
    {
        $this->authorize('update', $workflow);

        $validated = $request->validate([
            'name' => 'sometimes|string|max:255',
            'description' => 'nullable|string',
            'trigger' => 'sometimes|array',
            'steps' => 'sometimes|array',
        ]);

        DB::transaction(function () use ($validated, $workflow) {
            $workflow->update([
                'name' => $validated['name'] ?? $workflow->name,
                'description' => $validated['description'] ?? $workflow->description,
                'trigger' => $validated['trigger'] ?? $workflow->trigger,
            ]);

            if (isset($validated['steps'])) {
                $workflow->steps()->delete();
                foreach ($validated['steps'] as $index => $step) {
                    $workflow->steps()->create([
                        'tenant_id' => $workflow->tenant_id,
                        'type' => $step['type'],
                        'config' => $step['config'],
                        'delay' => $step['delay'] ?? null,
                        'position' => $index + 1,
                    ]);
                }
            }
        });

        return $this->success($workflow->fresh()->load('steps'), 'Workflow updated');
    }

    public function destroy(Workflow $workflow): JsonResponse
    {
        $this->authorize('delete', $workflow);
        $workflow->delete();
        return $this->success(null, 'Workflow deleted');
    }

    public function activate(Workflow $workflow): JsonResponse
    {
        $this->authorize('update', $workflow);

        if ($workflow->steps()->count() === 0) {
            return $this->error('Workflow must have at least one step', 422);
        }

        $workflow->update(['is_active' => true]);
        return $this->success($workflow, 'Workflow activated');
    }

    public function deactivate(Workflow $workflow): JsonResponse
    {
        $this->authorize('update', $workflow);
        $workflow->update(['is_active' => false]);
        return $this->success($workflow, 'Workflow deactivated');
    }

    public function enrollments(Workflow $workflow): JsonResponse
    {
        $this->authorize('view', $workflow);

        $enrollments = $workflow->enrollments()
            ->with('contact:id,first_name,last_name,email')
            ->orderBy('created_at', 'desc')
            ->paginate(50);

        return $this->paginated($enrollments);
    }

    public function unenroll(Workflow $workflow, WorkflowEnrollment $enrollment): JsonResponse
    {
        $this->authorize('update', $workflow);

        if ($enrollment->workflow_id !== $workflow->id) {
            return $this->error('Enrollment does not belong to this workflow', 422);
        }

        $this->workflowService->unenroll($enrollment);
        return $this->success(null, 'Contact unenrolled');
    }
}
