<?php

namespace App\Jobs;

use App\Models\BankAccount;
use App\Services\BankingService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Http;

class SyncPlaidTransactions implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $tries = 3;

    public function __construct(public BankAccount $bankAccount) {}

    public function handle(BankingService $bankingService): void
    {
        if (!$this->bankAccount->plaid_access_token) {
            return;
        }

        $response = Http::post(config('services.plaid.env') . '/transactions/get', [
            'client_id' => config('services.plaid.client_id'),
            'secret' => config('services.plaid.secret'),
            'access_token' => $this->bankAccount->plaid_access_token,
            'start_date' => now()->subDays(30)->toDateString(),
            'end_date' => now()->toDateString(),
        ]);

        if ($response->successful()) {
            $transactions = $response->json('transactions', []);
            $imported = $bankingService->importTransactions($this->bankAccount, $transactions);

            $this->bankAccount->update([
                'last_synced_at' => now(),
                'plaid_status' => 'connected',
            ]);
        }
    }
}
