<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class Activity extends BaseModel
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'tenant_id',
        'contact_id',
        'user_id',
        'activityable_type',
        'activityable_id',
        'type',
        'subject',
        'description',
        'outcome',
        'scheduled_at',
        'completed_at',
        'due_at',
        'duration_minutes',
        'is_completed',
        'metadata',
    ];

    protected $casts = [
        'scheduled_at' => 'datetime',
        'completed_at' => 'datetime',
        'due_at' => 'datetime',
        'duration_minutes' => 'integer',
        'is_completed' => 'boolean',
        'metadata' => 'array',
    ];

    const TYPE_CALL = 'call';
    const TYPE_EMAIL = 'email';
    const TYPE_MEETING = 'meeting';
    const TYPE_NOTE = 'note';
    const TYPE_TASK = 'task';
    const TYPE_SMS = 'sms';
    const TYPE_WHATSAPP = 'whatsapp';
    const TYPE_FOLLOW_UP = 'follow_up';

    const OUTCOME_COMPLETED = 'completed';
    const OUTCOME_NO_ANSWER = 'no_answer';
    const OUTCOME_LEFT_VOICEMAIL = 'left_voicemail';
    const OUTCOME_WRONG_NUMBER = 'wrong_number';
    const OUTCOME_INTERESTED = 'interested';
    const OUTCOME_NOT_INTERESTED = 'not_interested';
    const OUTCOME_CALLBACK = 'callback';

    protected static function booted(): void
    {
        static::creating(function ($activity) {
            if (empty($activity->user_id)) {
                $activity->user_id = auth()->id();
            }
        });
    }

    public function contact(): BelongsTo
    {
        return $this->belongsTo(Contact::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function activityable(): MorphTo
    {
        return $this->morphTo();
    }

    public function complete(string $outcome = null): void
    {
        $this->update([
            'is_completed' => true,
            'completed_at' => now(),
            'outcome' => $outcome ?? self::OUTCOME_COMPLETED,
        ]);
    }

    public function reopen(): void
    {
        $this->update([
            'is_completed' => false,
            'completed_at' => null,
            'outcome' => null,
        ]);
    }

    public function reschedule(\DateTime $dateTime): void
    {
        $this->update([
            'scheduled_at' => $dateTime,
            'is_completed' => false,
            'completed_at' => null,
        ]);
    }

    public function isOverdue(): bool
    {
        if ($this->is_completed) {
            return false;
        }

        $dueDate = $this->due_at ?? $this->scheduled_at;
        return $dueDate && $dueDate->isPast();
    }

    public function scopeUpcoming($query)
    {
        return $query->where('is_completed', false)
            ->where(function ($q) {
                $q->where('scheduled_at', '>=', now())
                  ->orWhere('due_at', '>=', now());
            })
            ->orderBy('scheduled_at');
    }

    public function scopeOverdue($query)
    {
        return $query->where('is_completed', false)
            ->where(function ($q) {
                $q->where('scheduled_at', '<', now())
                  ->orWhere('due_at', '<', now());
            });
    }

    public function scopeOfType($query, string $type)
    {
        return $query->where('type', $type);
    }

    public static function log(Contact $contact, string $type, array $data = []): self
    {
        return static::create([
            'tenant_id' => $contact->tenant_id,
            'contact_id' => $contact->id,
            'type' => $type,
            'subject' => $data['subject'] ?? null,
            'description' => $data['description'] ?? null,
            'outcome' => $data['outcome'] ?? null,
            'scheduled_at' => $data['scheduled_at'] ?? null,
            'completed_at' => $data['completed_at'] ?? now(),
            'is_completed' => $data['is_completed'] ?? true,
            'metadata' => $data['metadata'] ?? null,
            'activityable_type' => $data['activityable_type'] ?? null,
            'activityable_id' => $data['activityable_id'] ?? null,
        ]);
    }
}
