<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class BankRule extends BaseModel
{
    use HasFactory;

    protected $fillable = [
        'tenant_id',
        'name',
        'conditions',
        'account_id',
        'payee',
        'description',
        'is_active',
        'priority',
        'auto_categorize',
        'times_applied',
    ];

    protected $casts = [
        'conditions' => 'array',
        'is_active' => 'boolean',
        'auto_categorize' => 'boolean',
        'priority' => 'integer',
        'times_applied' => 'integer',
    ];

    public function account(): BelongsTo
    {
        return $this->belongsTo(ChartOfAccount::class, 'account_id');
    }

    public function matches(BankTransaction $transaction): bool
    {
        if (empty($this->conditions)) {
            return false;
        }

        foreach ($this->conditions as $condition) {
            if (!$this->evaluateCondition($condition, $transaction)) {
                return false;
            }
        }

        return true;
    }

    protected function evaluateCondition(array $condition, BankTransaction $transaction): bool
    {
        $field = $condition['field'] ?? null;
        $operator = $condition['operator'] ?? 'contains';
        $value = $condition['value'] ?? '';

        $transactionValue = match ($field) {
            'description' => strtolower($transaction->description ?? ''),
            'payee' => strtolower($transaction->payee ?? ''),
            'amount' => $transaction->amount,
            'type' => $transaction->type,
            'reference' => strtolower($transaction->reference ?? ''),
            default => '',
        };

        $value = is_string($value) ? strtolower($value) : $value;

        return match ($operator) {
            'contains' => str_contains($transactionValue, $value),
            'equals' => $transactionValue == $value,
            'starts_with' => str_starts_with($transactionValue, $value),
            'ends_with' => str_ends_with($transactionValue, $value),
            'greater_than' => $transactionValue > $value,
            'less_than' => $transactionValue < $value,
            'between' => $transactionValue >= ($condition['min'] ?? 0) && $transactionValue <= ($condition['max'] ?? PHP_INT_MAX),
            default => false,
        };
    }

    public function apply(BankTransaction $transaction): void
    {
        if ($this->account_id) {
            $transaction->account_id = $this->account_id;
        }
        if ($this->payee) {
            $transaction->payee = $this->payee;
        }
        if ($this->description) {
            $transaction->description = $this->description;
        }
        $transaction->save();

        $this->increment('times_applied');
    }

    public static function findMatchingRule($tenantId, BankTransaction $transaction): ?self
    {
        return static::where('tenant_id', $tenantId)
            ->where('is_active', true)
            ->orderBy('priority')
            ->get()
            ->first(fn ($rule) => $rule->matches($transaction));
    }
}
