<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class BankTransaction extends BaseModel
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'tenant_id',
        'bank_account_id',
        'account_id',
        'type',
        'transaction_date',
        'amount',
        'description',
        'reference',
        'payee',
        'category',
        'is_reconciled',
        'reconciled_at',
        'is_excluded',
        'matchable_type',
        'matchable_id',
        'plaid_transaction_id',
        'plaid_data',
        'notes',
    ];

    protected $casts = [
        'transaction_date' => 'date',
        'amount' => 'decimal:3',
        'is_reconciled' => 'boolean',
        'reconciled_at' => 'datetime',
        'is_excluded' => 'boolean',
        'plaid_data' => 'array',
    ];

    const TYPE_CREDIT = 'credit';
    const TYPE_DEBIT = 'debit';

    public function bankAccount(): BelongsTo
    {
        return $this->belongsTo(BankAccount::class);
    }

    public function account(): BelongsTo
    {
        return $this->belongsTo(ChartOfAccount::class, 'account_id');
    }

    public function matchable(): MorphTo
    {
        return $this->morphTo();
    }

    public function match($matchable): void
    {
        $this->update([
            'matchable_type' => get_class($matchable),
            'matchable_id' => $matchable->id,
            'is_reconciled' => true,
            'reconciled_at' => now(),
        ]);
    }

    public function unmatch(): void
    {
        $this->update([
            'matchable_type' => null,
            'matchable_id' => null,
            'is_reconciled' => false,
            'reconciled_at' => null,
        ]);
    }

    public function exclude(): void
    {
        $this->update([
            'is_excluded' => true,
            'is_reconciled' => true,
            'reconciled_at' => now(),
        ]);
    }

    public function include(): void
    {
        $this->update([
            'is_excluded' => false,
            'is_reconciled' => false,
            'reconciled_at' => null,
        ]);
    }

    public function categorize(ChartOfAccount $account): void
    {
        $this->update(['account_id' => $account->id]);
    }

    public function isMatched(): bool
    {
        return !empty($this->matchable_type) && !empty($this->matchable_id);
    }

    public function scopeUnreconciled($query)
    {
        return $query->where('is_reconciled', false)->where('is_excluded', false);
    }

    public function scopeCredits($query)
    {
        return $query->where('type', self::TYPE_CREDIT);
    }

    public function scopeDebits($query)
    {
        return $query->where('type', self::TYPE_DEBIT);
    }

    public function scopeDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('transaction_date', [$startDate, $endDate]);
    }
}
