<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class Bill extends BaseModel
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'tenant_id',
        'vendor_id',
        'bill_number',
        'vendor_invoice_number',
        'status',
        'bill_date',
        'due_date',
        'received_date',
        'currency',
        'exchange_rate',
        'subtotal',
        'discount_type',
        'discount_value',
        'discount_amount',
        'tax_amount',
        'total',
        'amount_paid',
        'notes',
        'terms',
        'created_by',
    ];

    protected $casts = [
        'bill_date' => 'date',
        'due_date' => 'date',
        'received_date' => 'date',
        'exchange_rate' => 'decimal:6',
        'subtotal' => 'decimal:3',
        'discount_value' => 'decimal:2',
        'discount_amount' => 'decimal:3',
        'tax_amount' => 'decimal:3',
        'total' => 'decimal:3',
        'amount_paid' => 'decimal:3',
    ];

    const STATUS_DRAFT = 'draft';
    const STATUS_RECEIVED = 'received';
    const STATUS_PARTIAL = 'partial';
    const STATUS_PAID = 'paid';
    const STATUS_OVERDUE = 'overdue';
    const STATUS_CANCELLED = 'cancelled';

    protected static function booted(): void
    {
        static::creating(function ($bill) {
            if (empty($bill->bill_number)) {
                $bill->bill_number = static::generateNumber($bill->tenant_id);
            }
            if (empty($bill->status)) {
                $bill->status = self::STATUS_DRAFT;
            }
        });
    }

    public static function generateNumber($tenantId): string
    {
        $latest = static::where('tenant_id', $tenantId)
            ->whereYear('created_at', now()->year)
            ->latest('id')
            ->first();

        $number = $latest ? (int) substr($latest->bill_number, -5) + 1 : 1;
        return 'BILL-' . now()->format('Y') . '-' . str_pad($number, 5, '0', STR_PAD_LEFT);
    }

    public function vendor(): BelongsTo
    {
        return $this->belongsTo(Vendor::class);
    }

    public function items(): HasMany
    {
        return $this->hasMany(BillItem::class)->orderBy('position');
    }

    public function payments(): HasMany
    {
        return $this->hasMany(BillPayment::class);
    }

    public function expenses(): HasMany
    {
        return $this->hasMany(Expense::class);
    }

    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function calculateTotals(): void
    {
        $this->subtotal = $this->items->sum('subtotal');
        
        if ($this->discount_type === 'percentage') {
            $this->discount_amount = $this->subtotal * ($this->discount_value / 100);
        } else {
            $this->discount_amount = $this->discount_value ?? 0;
        }
        
        $this->tax_amount = $this->items->sum('tax_amount');
        $this->total = $this->subtotal - $this->discount_amount + $this->tax_amount;
    }

    public function getAmountDue(): float
    {
        return $this->total - ($this->amount_paid ?? 0);
    }

    public function updatePaymentStatus(): void
    {
        $this->amount_paid = $this->payments()->where('status', 'completed')->sum('amount');
        
        if ($this->amount_paid >= $this->total) {
            $this->status = self::STATUS_PAID;
        } elseif ($this->amount_paid > 0) {
            $this->status = self::STATUS_PARTIAL;
        } elseif ($this->due_date && $this->due_date->isPast()) {
            $this->status = self::STATUS_OVERDUE;
        }
        
        $this->save();
    }

    public function markAsReceived(): void
    {
        $this->update([
            'status' => self::STATUS_RECEIVED,
            'received_date' => now(),
        ]);
    }

    public function isPaid(): bool
    {
        return $this->status === self::STATUS_PAID;
    }

    public function isOverdue(): bool
    {
        return $this->due_date && $this->due_date->isPast() && !$this->isPaid();
    }

    public function void(): void
    {
        $this->update(['status' => self::STATUS_CANCELLED]);
    }
}
