<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class BillPayment extends BaseModel
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'tenant_id',
        'bill_id',
        'vendor_id',
        'bank_account_id',
        'payment_number',
        'payment_date',
        'amount',
        'currency',
        'exchange_rate',
        'payment_method',
        'reference',
        'description',
        'status',
        'created_by',
    ];

    protected $casts = [
        'payment_date' => 'date',
        'amount' => 'decimal:3',
        'exchange_rate' => 'decimal:6',
    ];

    const STATUS_PENDING = 'pending';
    const STATUS_COMPLETED = 'completed';
    const STATUS_FAILED = 'failed';
    const STATUS_CANCELLED = 'cancelled';

    protected static function booted(): void
    {
        static::creating(function ($payment) {
            if (empty($payment->payment_number)) {
                $payment->payment_number = static::generateNumber($payment->tenant_id);
            }
        });

        static::created(function ($payment) {
            if ($payment->bill_id && $payment->status === self::STATUS_COMPLETED) {
                $payment->bill->updatePaymentStatus();
            }
        });
    }

    public static function generateNumber($tenantId): string
    {
        $latest = static::where('tenant_id', $tenantId)
            ->whereYear('created_at', now()->year)
            ->latest('id')
            ->first();

        $number = $latest ? (int) substr($latest->payment_number, -5) + 1 : 1;
        return 'BP-' . now()->format('Y') . '-' . str_pad($number, 5, '0', STR_PAD_LEFT);
    }

    public function bill(): BelongsTo
    {
        return $this->belongsTo(Bill::class);
    }

    public function vendor(): BelongsTo
    {
        return $this->belongsTo(Vendor::class);
    }

    public function bankAccount(): BelongsTo
    {
        return $this->belongsTo(BankAccount::class);
    }

    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }
}
