<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;

class Currency extends BaseModel
{
    use HasFactory;

    protected $fillable = [
        'tenant_id',
        'code',
        'name',
        'symbol',
        'exchange_rate',
        'decimal_places',
        'symbol_position',
        'thousands_separator',
        'decimal_separator',
        'is_default',
        'is_active',
    ];

    protected $casts = [
        'exchange_rate' => 'decimal:6',
        'decimal_places' => 'integer',
        'is_default' => 'boolean',
        'is_active' => 'boolean',
    ];

    const POSITION_BEFORE = 'before';
    const POSITION_AFTER = 'after';

    public static function getDefault($tenantId)
    {
        return static::where('tenant_id', $tenantId)
            ->where('is_default', true)
            ->first();
    }

    public function format(float $amount): string
    {
        $formatted = number_format(
            $amount,
            $this->decimal_places ?? 2,
            $this->decimal_separator ?? '.',
            $this->thousands_separator ?? ','
        );

        if ($this->symbol_position === self::POSITION_AFTER) {
            return $formatted . ' ' . $this->symbol;
        }

        return $this->symbol . $formatted;
    }

    public function convert(float $amount, Currency $toCurrency): float
    {
        if ($this->code === $toCurrency->code) {
            return $amount;
        }

        // Convert to base currency first, then to target
        $baseAmount = $amount / $this->exchange_rate;
        return $baseAmount * $toCurrency->exchange_rate;
    }

    public static function seedDefaults($tenantId): void
    {
        $currencies = [
            ['code' => 'BHD', 'name' => 'Bahraini Dinar', 'symbol' => 'BD', 'decimal_places' => 3, 'exchange_rate' => 1, 'is_default' => true],
            ['code' => 'USD', 'name' => 'US Dollar', 'symbol' => '$', 'decimal_places' => 2, 'exchange_rate' => 2.6525],
            ['code' => 'EUR', 'name' => 'Euro', 'symbol' => '€', 'decimal_places' => 2, 'exchange_rate' => 2.8850],
            ['code' => 'GBP', 'name' => 'British Pound', 'symbol' => '£', 'decimal_places' => 2, 'exchange_rate' => 3.3650],
            ['code' => 'SAR', 'name' => 'Saudi Riyal', 'symbol' => 'SR', 'decimal_places' => 2, 'exchange_rate' => 0.7073],
            ['code' => 'AED', 'name' => 'UAE Dirham', 'symbol' => 'AED', 'decimal_places' => 2, 'exchange_rate' => 0.7223],
        ];

        foreach ($currencies as $currency) {
            static::firstOrCreate(
                ['tenant_id' => $tenantId, 'code' => $currency['code']],
                array_merge($currency, ['tenant_id' => $tenantId, 'is_active' => true])
            );
        }
    }
}
