<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;

class EmailTemplate extends BaseModel
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'tenant_id',
        'name',
        'subject',
        'body',
        'type',
        'category',
        'variables',
        'is_system',
        'is_active',
    ];

    protected $casts = [
        'variables' => 'array',
        'is_system' => 'boolean',
        'is_active' => 'boolean',
    ];

    const TYPE_PLAIN = 'plain';
    const TYPE_HTML = 'html';
    const TYPE_MJML = 'mjml';

    public function render(array $data = []): array
    {
        $subject = $this->replaceVariables($this->subject, $data);
        $body = $this->replaceVariables($this->body, $data);

        return [
            'subject' => $subject,
            'body' => $body,
        ];
    }

    protected function replaceVariables(string $content, array $data): string
    {
        foreach ($data as $key => $value) {
            if (is_string($value) || is_numeric($value)) {
                $content = str_replace("{{" . $key . "}}", $value, $content);
                $content = str_replace("{{ " . $key . " }}", $value, $content);
            }
        }

        // Remove any unreplaced variables
        $content = preg_replace('/\{\{\s*\w+\s*\}\}/', '', $content);

        return $content;
    }

    public function getAvailableVariables(): array
    {
        return $this->variables ?? [];
    }

    public function duplicate(): self
    {
        $copy = $this->replicate();
        $copy->name = $this->name . ' (Copy)';
        $copy->is_system = false;
        $copy->save();

        return $copy;
    }

    public static function getByCategory($tenantId, string $category)
    {
        return static::where('tenant_id', $tenantId)
            ->where('category', $category)
            ->where('is_active', true)
            ->get();
    }
}
