<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class Estimate extends BaseModel
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'tenant_id',
        'contact_id',
        'estimate_number',
        'reference',
        'status',
        'estimate_date',
        'expiry_date',
        'currency',
        'exchange_rate',
        'billing_address',
        'subtotal',
        'discount_type',
        'discount_value',
        'discount_amount',
        'tax_amount',
        'total',
        'notes',
        'terms',
        'footer',
        'sent_at',
        'viewed_at',
        'accepted_at',
        'declined_at',
        'declined_reason',
        'converted_to_invoice_id',
        'created_by',
    ];

    protected $casts = [
        'estimate_date' => 'date',
        'expiry_date' => 'date',
        'billing_address' => 'array',
        'exchange_rate' => 'decimal:6',
        'subtotal' => 'decimal:3',
        'discount_value' => 'decimal:2',
        'discount_amount' => 'decimal:3',
        'tax_amount' => 'decimal:3',
        'total' => 'decimal:3',
        'sent_at' => 'datetime',
        'viewed_at' => 'datetime',
        'accepted_at' => 'datetime',
        'declined_at' => 'datetime',
    ];

    const STATUS_DRAFT = 'draft';
    const STATUS_SENT = 'sent';
    const STATUS_VIEWED = 'viewed';
    const STATUS_ACCEPTED = 'accepted';
    const STATUS_DECLINED = 'declined';
    const STATUS_EXPIRED = 'expired';
    const STATUS_CONVERTED = 'converted';

    protected static function booted(): void
    {
        static::creating(function ($estimate) {
            if (empty($estimate->estimate_number)) {
                $estimate->estimate_number = static::generateNumber($estimate->tenant_id);
            }
            if (empty($estimate->status)) {
                $estimate->status = self::STATUS_DRAFT;
            }
        });
    }

    public static function generateNumber($tenantId): string
    {
        $latest = static::where('tenant_id', $tenantId)
            ->whereYear('created_at', now()->year)
            ->latest('id')
            ->first();

        $number = $latest ? (int) substr($latest->estimate_number, -5) + 1 : 1;
        return 'EST-' . now()->format('Y') . '-' . str_pad($number, 5, '0', STR_PAD_LEFT);
    }

    public function contact(): BelongsTo
    {
        return $this->belongsTo(Contact::class);
    }

    public function items(): HasMany
    {
        return $this->hasMany(EstimateItem::class)->orderBy('position');
    }

    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function convertedInvoice(): BelongsTo
    {
        return $this->belongsTo(Invoice::class, 'converted_to_invoice_id');
    }

    public function calculateTotals(): void
    {
        $this->subtotal = $this->items->sum('subtotal');
        
        if ($this->discount_type === 'percentage') {
            $this->discount_amount = $this->subtotal * ($this->discount_value / 100);
        } else {
            $this->discount_amount = $this->discount_value ?? 0;
        }
        
        $this->tax_amount = $this->items->sum('tax_amount');
        $this->total = $this->subtotal - $this->discount_amount + $this->tax_amount;
    }

    public function markAsSent(): void
    {
        $this->update([
            'status' => self::STATUS_SENT,
            'sent_at' => now(),
        ]);
    }

    public function markAsViewed(): void
    {
        if ($this->status === self::STATUS_SENT) {
            $this->update([
                'status' => self::STATUS_VIEWED,
                'viewed_at' => now(),
            ]);
        }
    }

    public function accept(): void
    {
        $this->update([
            'status' => self::STATUS_ACCEPTED,
            'accepted_at' => now(),
        ]);
    }

    public function decline(string $reason = null): void
    {
        $this->update([
            'status' => self::STATUS_DECLINED,
            'declined_at' => now(),
            'declined_reason' => $reason,
        ]);
    }

    public function convertToInvoice(): Invoice
    {
        $invoice = Invoice::create([
            'tenant_id' => $this->tenant_id,
            'contact_id' => $this->contact_id,
            'reference' => $this->reference,
            'invoice_date' => now(),
            'due_date' => now()->addDays(30),
            'currency' => $this->currency,
            'exchange_rate' => $this->exchange_rate,
            'billing_address' => $this->billing_address,
            'subtotal' => $this->subtotal,
            'discount_type' => $this->discount_type,
            'discount_value' => $this->discount_value,
            'discount_amount' => $this->discount_amount,
            'tax_amount' => $this->tax_amount,
            'total' => $this->total,
            'notes' => $this->notes,
            'terms' => $this->terms,
            'footer' => $this->footer,
            'created_by' => auth()->id(),
        ]);

        foreach ($this->items as $item) {
            $invoice->items()->create($item->only([
                'product_id', 'name', 'description', 'quantity', 'unit_price',
                'discount_type', 'discount_value', 'discount_amount',
                'tax_rate_id', 'tax_amount', 'subtotal', 'total', 'position'
            ]));
        }

        $this->update([
            'status' => self::STATUS_CONVERTED,
            'converted_to_invoice_id' => $invoice->id,
        ]);

        return $invoice;
    }

    public function isExpired(): bool
    {
        return $this->expiry_date && $this->expiry_date->isPast() && 
               !in_array($this->status, [self::STATUS_ACCEPTED, self::STATUS_CONVERTED]);
    }
}
