<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use App\Traits\BelongsToTenant;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;

class Opportunity extends Model
{
    use HasFactory, SoftDeletes, BelongsToTenant, LogsActivity;

    protected $fillable = [
        'tenant_id',
        'pipeline_id',
        'stage_id',
        'contact_id',
        'company_id',
        'owner_id',
        'title',
        'description',
        'value',
        'currency',
        'expected_close_date',
        'actual_close_date',
        'probability',
        'source',
        'status',
        'lost_reason',
        'package',
        'business_activity',
        'legal_form',
        'position',
        'custom_fields',
    ];

    protected $casts = [
        'value' => 'decimal:2',
        'probability' => 'decimal:2',
        'expected_close_date' => 'date',
        'actual_close_date' => 'date',
        'custom_fields' => 'array',
    ];

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly(['title', 'value', 'stage_id', 'owner_id', 'status'])
            ->logOnlyDirty();
    }

    // Relationships
    public function tenant(): BelongsTo
    {
        return $this->belongsTo(Tenant::class);
    }

    public function pipeline(): BelongsTo
    {
        return $this->belongsTo(Pipeline::class);
    }

    public function stage(): BelongsTo
    {
        return $this->belongsTo(PipelineStage::class, 'stage_id');
    }

    public function contact(): BelongsTo
    {
        return $this->belongsTo(Contact::class);
    }

    public function company(): BelongsTo
    {
        return $this->belongsTo(Contact::class, 'company_id');
    }

    public function owner(): BelongsTo
    {
        return $this->belongsTo(User::class, 'owner_id');
    }

    public function products(): HasMany
    {
        return $this->hasMany(OpportunityProduct::class);
    }

    public function activities(): MorphMany
    {
        return $this->morphMany(Activity::class, 'activitable');
    }

    public function invoices(): HasMany
    {
        return $this->hasMany(Invoice::class);
    }

    public function project(): HasMany
    {
        return $this->hasMany(Project::class);
    }

    // Scopes
    public function scopeOpen($query)
    {
        return $query->where('status', 'open');
    }

    public function scopeWon($query)
    {
        return $query->where('status', 'won');
    }

    public function scopeLost($query)
    {
        return $query->where('status', 'lost');
    }

    // Helpers
    public function isOpen(): bool
    {
        return $this->status === 'open';
    }

    public function isWon(): bool
    {
        return $this->status === 'won';
    }

    public function isLost(): bool
    {
        return $this->status === 'lost';
    }

    public function markAsWon(): void
    {
        $wonStage = $this->pipeline->stages()->where('is_won', true)->first();
        
        $this->update([
            'status' => 'won',
            'stage_id' => $wonStage?->id ?? $this->stage_id,
            'actual_close_date' => now(),
            'probability' => 100,
        ]);

        if ($this->contact) {
            $this->contact->addToLifetimeValue($this->value);
        }
    }

    public function markAsLost(string $reason = null): void
    {
        $lostStage = $this->pipeline->stages()->where('is_lost', true)->first();
        
        $this->update([
            'status' => 'lost',
            'stage_id' => $lostStage?->id ?? $this->stage_id,
            'actual_close_date' => now(),
            'lost_reason' => $reason,
            'probability' => 0,
        ]);
    }

    public function moveToStage(PipelineStage $stage): void
    {
        $oldStage = $this->stage;
        
        $this->update([
            'stage_id' => $stage->id,
            'probability' => $stage->probability,
        ]);

        // Log stage change activity
        $this->activities()->create([
            'tenant_id' => $this->tenant_id,
            'user_id' => auth()->id(),
            'type' => 'stage_change',
            'title' => "Moved from {$oldStage->name} to {$stage->name}",
            'metadata' => [
                'old_stage_id' => $oldStage->id,
                'new_stage_id' => $stage->id,
            ],
        ]);
    }

    public function getWeightedValue(): float
    {
        return $this->value * ($this->probability / 100);
    }
}
