<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class Payment extends BaseModel
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'tenant_id',
        'contact_id',
        'invoice_id',
        'bank_account_id',
        'payment_number',
        'payment_date',
        'amount',
        'currency',
        'exchange_rate',
        'payment_method',
        'reference',
        'description',
        'transaction_id',
        'gateway_response',
        'status',
        'refunded_amount',
        'refunded_at',
        'created_by',
    ];

    protected $casts = [
        'payment_date' => 'date',
        'amount' => 'decimal:3',
        'exchange_rate' => 'decimal:6',
        'gateway_response' => 'array',
        'refunded_amount' => 'decimal:3',
        'refunded_at' => 'datetime',
    ];

    const STATUS_PENDING = 'pending';
    const STATUS_COMPLETED = 'completed';
    const STATUS_FAILED = 'failed';
    const STATUS_REFUNDED = 'refunded';
    const STATUS_PARTIALLY_REFUNDED = 'partially_refunded';

    const METHOD_CASH = 'cash';
    const METHOD_BANK_TRANSFER = 'bank_transfer';
    const METHOD_CREDIT_CARD = 'credit_card';
    const METHOD_CHEQUE = 'cheque';
    const METHOD_ONLINE = 'online';

    protected static function booted(): void
    {
        static::creating(function ($payment) {
            if (empty($payment->payment_number)) {
                $payment->payment_number = static::generateNumber($payment->tenant_id);
            }
        });

        static::created(function ($payment) {
            if ($payment->invoice_id && $payment->status === self::STATUS_COMPLETED) {
                $payment->invoice->updatePaymentStatus();
            }
        });
    }

    public static function generateNumber($tenantId): string
    {
        $latest = static::where('tenant_id', $tenantId)
            ->whereYear('created_at', now()->year)
            ->latest('id')
            ->first();

        $number = $latest ? (int) substr($latest->payment_number, -5) + 1 : 1;
        return 'PAY-' . now()->format('Y') . '-' . str_pad($number, 5, '0', STR_PAD_LEFT);
    }

    public function contact(): BelongsTo
    {
        return $this->belongsTo(Contact::class);
    }

    public function invoice(): BelongsTo
    {
        return $this->belongsTo(Invoice::class);
    }

    public function bankAccount(): BelongsTo
    {
        return $this->belongsTo(BankAccount::class);
    }

    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function refund(float $amount = null): void
    {
        $refundAmount = $amount ?? $this->amount;
        
        if ($refundAmount > ($this->amount - $this->refunded_amount)) {
            throw new \Exception('Refund amount exceeds available amount');
        }

        $this->refunded_amount = ($this->refunded_amount ?? 0) + $refundAmount;
        $this->refunded_at = now();
        
        if ($this->refunded_amount >= $this->amount) {
            $this->status = self::STATUS_REFUNDED;
        } else {
            $this->status = self::STATUS_PARTIALLY_REFUNDED;
        }
        
        $this->save();

        if ($this->invoice_id) {
            $this->invoice->updatePaymentStatus();
        }
    }

    public function getAvailableRefundAmount(): float
    {
        return $this->amount - ($this->refunded_amount ?? 0);
    }
}
