<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class RecurringInvoice extends BaseModel
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'tenant_id',
        'contact_id',
        'status',
        'frequency',
        'interval',
        'start_date',
        'end_date',
        'next_invoice_date',
        'last_invoice_date',
        'invoices_generated',
        'max_invoices',
        'auto_send',
        'days_before_due',
        'currency',
        'billing_address',
        'invoice_data',
        'created_by',
    ];

    protected $casts = [
        'start_date' => 'date',
        'end_date' => 'date',
        'next_invoice_date' => 'date',
        'last_invoice_date' => 'date',
        'invoices_generated' => 'integer',
        'max_invoices' => 'integer',
        'auto_send' => 'boolean',
        'days_before_due' => 'integer',
        'billing_address' => 'array',
        'invoice_data' => 'array',
    ];

    const STATUS_DRAFT = 'draft';
    const STATUS_ACTIVE = 'active';
    const STATUS_PAUSED = 'paused';
    const STATUS_COMPLETED = 'completed';
    const STATUS_CANCELLED = 'cancelled';

    const FREQUENCY_DAILY = 'daily';
    const FREQUENCY_WEEKLY = 'weekly';
    const FREQUENCY_MONTHLY = 'monthly';
    const FREQUENCY_QUARTERLY = 'quarterly';
    const FREQUENCY_YEARLY = 'yearly';

    public function contact(): BelongsTo
    {
        return $this->belongsTo(Contact::class);
    }

    public function invoices(): HasMany
    {
        return $this->hasMany(Invoice::class);
    }

    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function activate(): void
    {
        $this->update([
            'status' => self::STATUS_ACTIVE,
            'next_invoice_date' => $this->calculateNextDate($this->start_date ?? now()),
        ]);
    }

    public function pause(): void
    {
        $this->update(['status' => self::STATUS_PAUSED]);
    }

    public function resume(): void
    {
        $this->update([
            'status' => self::STATUS_ACTIVE,
            'next_invoice_date' => $this->calculateNextDate(now()),
        ]);
    }

    public function cancel(): void
    {
        $this->update(['status' => self::STATUS_CANCELLED]);
    }

    public function generateInvoice(): ?Invoice
    {
        if (!$this->canGenerate()) {
            return null;
        }

        $invoiceData = $this->invoice_data;
        
        $invoice = Invoice::create([
            'tenant_id' => $this->tenant_id,
            'contact_id' => $this->contact_id,
            'recurring_invoice_id' => $this->id,
            'invoice_date' => now(),
            'due_date' => now()->addDays($this->days_before_due ?? 30),
            'currency' => $this->currency,
            'billing_address' => $this->billing_address,
            'subtotal' => $invoiceData['subtotal'] ?? 0,
            'discount_type' => $invoiceData['discount_type'] ?? null,
            'discount_value' => $invoiceData['discount_value'] ?? 0,
            'discount_amount' => $invoiceData['discount_amount'] ?? 0,
            'tax_amount' => $invoiceData['tax_amount'] ?? 0,
            'total' => $invoiceData['total'] ?? 0,
            'notes' => $invoiceData['notes'] ?? null,
            'terms' => $invoiceData['terms'] ?? null,
            'created_by' => $this->created_by,
        ]);

        if (!empty($invoiceData['items'])) {
            foreach ($invoiceData['items'] as $item) {
                $invoice->items()->create($item);
            }
        }

        $this->update([
            'invoices_generated' => $this->invoices_generated + 1,
            'last_invoice_date' => now(),
            'next_invoice_date' => $this->calculateNextDate(now()),
        ]);

        if ($this->isCompleted()) {
            $this->update(['status' => self::STATUS_COMPLETED]);
        }

        if ($this->auto_send) {
            // Dispatch job to send invoice
            // SendInvoiceEmail::dispatch($invoice);
        }

        return $invoice;
    }

    public function canGenerate(): bool
    {
        if ($this->status !== self::STATUS_ACTIVE) {
            return false;
        }

        if ($this->end_date && $this->end_date->isPast()) {
            return false;
        }

        if ($this->max_invoices && $this->invoices_generated >= $this->max_invoices) {
            return false;
        }

        return true;
    }

    public function isCompleted(): bool
    {
        if ($this->end_date && $this->end_date->isPast()) {
            return true;
        }

        if ($this->max_invoices && $this->invoices_generated >= $this->max_invoices) {
            return true;
        }

        return false;
    }

    public function calculateNextDate($fromDate): \Carbon\Carbon
    {
        $date = \Carbon\Carbon::parse($fromDate);
        $interval = $this->interval ?? 1;

        return match ($this->frequency) {
            self::FREQUENCY_DAILY => $date->addDays($interval),
            self::FREQUENCY_WEEKLY => $date->addWeeks($interval),
            self::FREQUENCY_MONTHLY => $date->addMonths($interval),
            self::FREQUENCY_QUARTERLY => $date->addQuarters($interval),
            self::FREQUENCY_YEARLY => $date->addYears($interval),
            default => $date->addMonths($interval),
        };
    }
}
