<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class SmsMessage extends BaseModel
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'tenant_id',
        'contact_id',
        'campaign_id',
        'direction',
        'from_number',
        'to_number',
        'message',
        'segments',
        'status',
        'provider_id',
        'provider_response',
        'cost',
        'sent_at',
        'delivered_at',
        'failed_at',
        'error_message',
    ];

    protected $casts = [
        'segments' => 'integer',
        'provider_response' => 'array',
        'cost' => 'decimal:4',
        'sent_at' => 'datetime',
        'delivered_at' => 'datetime',
        'failed_at' => 'datetime',
    ];

    const DIRECTION_INBOUND = 'inbound';
    const DIRECTION_OUTBOUND = 'outbound';

    const STATUS_QUEUED = 'queued';
    const STATUS_SENT = 'sent';
    const STATUS_DELIVERED = 'delivered';
    const STATUS_FAILED = 'failed';
    const STATUS_RECEIVED = 'received';

    public function contact(): BelongsTo
    {
        return $this->belongsTo(Contact::class);
    }

    public function campaign(): BelongsTo
    {
        return $this->belongsTo(Campaign::class);
    }

    public function markAsSent(string $providerId = null): void
    {
        $this->update([
            'status' => self::STATUS_SENT,
            'sent_at' => now(),
            'provider_id' => $providerId,
        ]);
    }

    public function markAsDelivered(): void
    {
        $this->update([
            'status' => self::STATUS_DELIVERED,
            'delivered_at' => now(),
        ]);
    }

    public function markAsFailed(string $reason = null): void
    {
        $this->update([
            'status' => self::STATUS_FAILED,
            'failed_at' => now(),
            'error_message' => $reason,
        ]);
    }

    public static function calculateSegments(string $message): int
    {
        $length = mb_strlen($message);
        
        // Check if message contains non-GSM characters
        $isUnicode = preg_match('/[^\x00-\x7F]/', $message);
        
        if ($isUnicode) {
            // Unicode: 70 chars per segment, 67 for multi-part
            return $length <= 70 ? 1 : ceil($length / 67);
        }
        
        // GSM-7: 160 chars per segment, 153 for multi-part
        return $length <= 160 ? 1 : ceil($length / 153);
    }

    public function scopeOutbound($query)
    {
        return $query->where('direction', self::DIRECTION_OUTBOUND);
    }

    public function scopeInbound($query)
    {
        return $query->where('direction', self::DIRECTION_INBOUND);
    }

    public function scopeConversation($query, string $phoneNumber)
    {
        return $query->where(function ($q) use ($phoneNumber) {
            $q->where('from_number', $phoneNumber)
              ->orWhere('to_number', $phoneNumber);
        })->orderBy('created_at');
    }
}
