<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class TaskComment extends BaseModel
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'tenant_id',
        'task_id',
        'user_id',
        'parent_id',
        'body',
        'is_internal',
        'mentions',
        'edited_at',
    ];

    protected $casts = [
        'is_internal' => 'boolean',
        'mentions' => 'array',
        'edited_at' => 'datetime',
    ];

    protected static function booted(): void
    {
        static::creating(function ($comment) {
            if (empty($comment->user_id)) {
                $comment->user_id = auth()->id();
            }
        });

        static::created(function ($comment) {
            // Parse mentions and notify users
            if (!empty($comment->mentions)) {
                foreach ($comment->mentions as $userId) {
                    // Dispatch notification
                }
            }
        });
    }

    public function task(): BelongsTo
    {
        return $this->belongsTo(Task::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function parent(): BelongsTo
    {
        return $this->belongsTo(TaskComment::class, 'parent_id');
    }

    public function replies(): HasMany
    {
        return $this->hasMany(TaskComment::class, 'parent_id')->orderBy('created_at');
    }

    public function isEdited(): bool
    {
        return !is_null($this->edited_at);
    }

    public function edit(string $body): void
    {
        $this->update([
            'body' => $body,
            'edited_at' => now(),
            'mentions' => $this->parseMentions($body),
        ]);
    }

    protected function parseMentions(string $body): array
    {
        preg_match_all('/@\[([^\]]+)\]\((\d+)\)/', $body, $matches);
        return array_map('intval', $matches[2] ?? []);
    }

    public function getMentionedUsers()
    {
        if (empty($this->mentions)) {
            return collect();
        }

        return User::whereIn('id', $this->mentions)->get();
    }
}
