<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class TaskDependency extends BaseModel
{
    use HasFactory;

    protected $fillable = [
        'tenant_id',
        'task_id',
        'depends_on_task_id',
        'type',
        'lag_days',
    ];

    protected $casts = [
        'lag_days' => 'integer',
    ];

    const TYPE_FINISH_TO_START = 'finish_to_start';
    const TYPE_START_TO_START = 'start_to_start';
    const TYPE_FINISH_TO_FINISH = 'finish_to_finish';
    const TYPE_START_TO_FINISH = 'start_to_finish';

    public function task(): BelongsTo
    {
        return $this->belongsTo(Task::class);
    }

    public function dependsOnTask(): BelongsTo
    {
        return $this->belongsTo(Task::class, 'depends_on_task_id');
    }

    public function isBlocking(): bool
    {
        $dependsOn = $this->dependsOnTask;
        
        if (!$dependsOn) {
            return false;
        }

        return match ($this->type) {
            self::TYPE_FINISH_TO_START => !$dependsOn->isCompleted(),
            self::TYPE_START_TO_START => !$dependsOn->start_date || $dependsOn->start_date->isFuture(),
            self::TYPE_FINISH_TO_FINISH => !$dependsOn->isCompleted(),
            self::TYPE_START_TO_FINISH => !$dependsOn->start_date || $dependsOn->start_date->isFuture(),
            default => false,
        };
    }

    public function getEarliestStartDate(): ?\Carbon\Carbon
    {
        $dependsOn = $this->dependsOnTask;
        
        if (!$dependsOn) {
            return null;
        }

        $baseDate = match ($this->type) {
            self::TYPE_FINISH_TO_START => $dependsOn->due_date ?? $dependsOn->completed_at,
            self::TYPE_START_TO_START => $dependsOn->start_date,
            self::TYPE_FINISH_TO_FINISH => $dependsOn->due_date ?? $dependsOn->completed_at,
            self::TYPE_START_TO_FINISH => $dependsOn->start_date,
            default => null,
        };

        if (!$baseDate) {
            return null;
        }

        return \Carbon\Carbon::parse($baseDate)->addDays($this->lag_days ?? 0);
    }
}
