<?php

namespace App\Notifications;

use App\Models\Invoice;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class InvoiceCreatedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(public Invoice $invoice) {}

    public function via($notifiable): array
    {
        return ['mail', 'database'];
    }

    public function toMail($notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('New Invoice #' . $this->invoice->invoice_number)
            ->greeting('Hello ' . $notifiable->first_name)
            ->line('A new invoice has been created for you.')
            ->line('Invoice Number: ' . $this->invoice->invoice_number)
            ->line('Amount: ' . number_format($this->invoice->total, 3) . ' ' . $this->invoice->currency)
            ->line('Due Date: ' . $this->invoice->due_date->format('M d, Y'))
            ->action('View Invoice', url('/invoices/' . $this->invoice->id))
            ->line('Thank you for your business!');
    }

    public function toArray($notifiable): array
    {
        return [
            'type' => 'invoice_created',
            'invoice_id' => $this->invoice->id,
            'invoice_number' => $this->invoice->invoice_number,
            'amount' => $this->invoice->total,
            'currency' => $this->invoice->currency,
        ];
    }
}
