<?php

namespace App\Services;

use App\Models\BankAccount;
use App\Models\BankTransaction;
use App\Models\Bill;
use App\Models\Invoice;
use App\Models\ChartOfAccount;
use Illuminate\Support\Facades\DB;

class BankingService
{
    public function importTransactions(BankAccount $account, array $transactions): int
    {
        $imported = 0;

        foreach ($transactions as $txn) {
            $exists = BankTransaction::where('bank_account_id', $account->id)
                ->where('external_id', $txn['transaction_id'])
                ->exists();

            if ($exists) continue;

            BankTransaction::create([
                'tenant_id' => $account->tenant_id,
                'bank_account_id' => $account->id,
                'external_id' => $txn['transaction_id'],
                'transaction_date' => $txn['date'],
                'description' => $txn['name'],
                'payee' => $txn['merchant_name'] ?? $txn['name'],
                'amount' => abs($txn['amount']),
                'type' => $txn['amount'] < 0 ? 'debit' : 'credit',
                'currency' => $txn['iso_currency_code'] ?? 'BHD',
                'category' => $txn['category'][0] ?? null,
                'status' => 'pending',
            ]);

            $imported++;
        }

        $this->updateAccountBalance($account);

        return $imported;
    }

    public function matchToInvoice(BankTransaction $transaction, int $invoiceId): void
    {
        $invoice = Invoice::findOrFail($invoiceId);

        if ($invoice->tenant_id !== $transaction->tenant_id) {
            throw new \Exception('Invoice does not belong to this tenant');
        }

        DB::transaction(function () use ($transaction, $invoice) {
            $transaction->update([
                'matched_type' => Invoice::class,
                'matched_id' => $invoice->id,
                'contact_id' => $invoice->contact_id,
                'status' => 'matched',
            ]);

            // Record payment on invoice
            app(InvoiceService::class)->recordPayment($invoice, [
                'amount' => $transaction->amount,
                'payment_date' => $transaction->transaction_date,
                'payment_method' => 'bank_transfer',
                'reference' => $transaction->external_id,
                'bank_transaction_id' => $transaction->id,
            ], auth()->id());
        });
    }

    public function matchToBill(BankTransaction $transaction, int $billId): void
    {
        $bill = Bill::findOrFail($billId);

        if ($bill->tenant_id !== $transaction->tenant_id) {
            throw new \Exception('Bill does not belong to this tenant');
        }

        DB::transaction(function () use ($transaction, $bill) {
            $transaction->update([
                'matched_type' => Bill::class,
                'matched_id' => $bill->id,
                'contact_id' => $bill->vendor_id,
                'status' => 'matched',
            ]);

            // Record payment on bill
            app(BillService::class)->recordPayment($bill, [
                'amount' => $transaction->amount,
                'payment_date' => $transaction->transaction_date,
                'payment_method' => 'bank_transfer',
                'reference' => $transaction->external_id,
                'bank_transaction_id' => $transaction->id,
            ], auth()->id());
        });
    }

    public function matchTransaction(BankTransaction $transaction, int $accountId): void
    {
        $account = ChartOfAccount::findOrFail($accountId);

        $transaction->update([
            'account_id' => $account->id,
            'status' => 'matched',
        ]);
    }

    public function getUnmatchedSuggestions(BankTransaction $transaction): array
    {
        $suggestions = [];

        if ($transaction->type === 'credit') {
            // Look for unpaid invoices
            $invoices = Invoice::where('tenant_id', $transaction->tenant_id)
                ->whereIn('status', ['sent', 'partial'])
                ->whereBetween('total', [$transaction->amount * 0.95, $transaction->amount * 1.05])
                ->limit(5)
                ->get();

            foreach ($invoices as $invoice) {
                $suggestions[] = [
                    'type' => 'invoice',
                    'id' => $invoice->id,
                    'reference' => $invoice->invoice_number,
                    'contact' => $invoice->contact->full_name ?? $invoice->contact->company_name,
                    'amount' => $invoice->total,
                    'date' => $invoice->invoice_date,
                ];
            }
        } else {
            // Look for unpaid bills
            $bills = Bill::where('tenant_id', $transaction->tenant_id)
                ->whereIn('status', ['received', 'partial'])
                ->whereBetween('total', [$transaction->amount * 0.95, $transaction->amount * 1.05])
                ->limit(5)
                ->get();

            foreach ($bills as $bill) {
                $suggestions[] = [
                    'type' => 'bill',
                    'id' => $bill->id,
                    'reference' => $bill->bill_number,
                    'vendor' => $bill->vendor->name,
                    'amount' => $bill->total,
                    'date' => $bill->bill_date,
                ];
            }
        }

        return $suggestions;
    }

    protected function updateAccountBalance(BankAccount $account): void
    {
        $balance = $account->opening_balance +
            $account->transactions()->where('type', 'credit')->sum('amount') -
            $account->transactions()->where('type', 'debit')->sum('amount');

        $account->update(['current_balance' => $balance]);
    }
}
