<?php

namespace App\Services;

use App\Jobs\SendCampaignEmail;
use App\Models\Campaign;
use App\Models\CampaignRecipient;
use App\Models\Contact;
use App\Models\Segment;

class CampaignService
{
    public function create(int $tenantId, array $data): Campaign
    {
        $campaign = Campaign::create([
            'tenant_id' => $tenantId,
            'name' => $data['name'],
            'type' => $data['type'],
            'subject' => $data['subject'] ?? null,
            'content' => $data['content'],
            'from_name' => $data['from_name'] ?? null,
            'from_email' => $data['from_email'] ?? null,
            'status' => 'draft',
        ]);

        if (!empty($data['segment_ids'])) {
            $campaign->segments()->sync($data['segment_ids']);
        }

        return $campaign;
    }

    public function send(Campaign $campaign): void
    {
        $campaign->update(['status' => 'sending', 'sent_at' => now()]);

        $recipients = $this->getRecipients($campaign);

        foreach ($recipients as $contact) {
            CampaignRecipient::create([
                'campaign_id' => $campaign->id,
                'contact_id' => $contact->id,
                'email' => $contact->email,
                'status' => 'pending',
            ]);
        }

        $campaign->update(['recipients_count' => $recipients->count()]);

        foreach ($campaign->recipients as $recipient) {
            SendCampaignEmail::dispatch($campaign, $recipient);
        }
    }

    public function schedule(Campaign $campaign, string $scheduledAt): void
    {
        $campaign->update([
            'status' => 'scheduled',
            'scheduled_at' => $scheduledAt,
        ]);
    }

    public function duplicate(Campaign $campaign): Campaign
    {
        $newCampaign = $campaign->replicate();
        $newCampaign->name = $campaign->name . ' (Copy)';
        $newCampaign->status = 'draft';
        $newCampaign->sent_at = null;
        $newCampaign->scheduled_at = null;
        $newCampaign->save();

        $newCampaign->segments()->sync($campaign->segments->pluck('id'));

        return $newCampaign;
    }

    public function getStats(Campaign $campaign): array
    {
        $recipients = $campaign->recipients;

        return [
            'total' => $recipients->count(),
            'delivered' => $recipients->where('status', 'delivered')->count(),
            'opened' => $recipients->where('opened_at', '!=', null)->count(),
            'clicked' => $recipients->where('clicked_at', '!=', null)->count(),
            'bounced' => $recipients->where('status', 'bounced')->count(),
            'unsubscribed' => $recipients->where('status', 'unsubscribed')->count(),
        ];
    }

    public function recordOpen(CampaignRecipient $recipient): void
    {
        if (!$recipient->opened_at) {
            $recipient->update([
                'opened_at' => now(),
                'open_count' => 1,
            ]);
        } else {
            $recipient->increment('open_count');
        }
    }

    public function recordClick(CampaignRecipient $recipient, string $url): void
    {
        if (!$recipient->clicked_at) {
            $recipient->update(['clicked_at' => now()]);
        }
        $recipient->increment('click_count');

        $recipient->clicks()->create([
            'url' => $url,
            'clicked_at' => now(),
        ]);
    }

    protected function getRecipients(Campaign $campaign)
    {
        $query = Contact::where('tenant_id', $campaign->tenant_id)
            ->whereNotNull('email')
            ->where('is_subscribed', true);

        if ($campaign->segments->isNotEmpty()) {
            $segmentIds = $campaign->segments->pluck('id');
            $query->whereHas('segments', fn($q) => $q->whereIn('segments.id', $segmentIds));
        }

        return $query->get();
    }
}
