<?php

namespace App\Services;

use App\Models\Contact;
use App\Models\Activity;
use App\Models\Tag;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\UploadedFile;
use Maatwebsite\Excel\Facades\Excel;

class ContactService
{
    public function create(array $data): Contact
    {
        return DB::transaction(function () use ($data) {
            $contact = Contact::create([
                'tenant_id' => $data['tenant_id'],
                'owner_id' => $data['owner_id'] ?? auth()->id(),
                'company_id' => $data['company_id'] ?? null,
                'type' => $data['type'] ?? 'person',
                'first_name' => $data['first_name'] ?? null,
                'last_name' => $data['last_name'] ?? null,
                'company_name' => $data['company_name'] ?? null,
                'email' => $data['email'] ?? null,
                'phone' => $data['phone'] ?? null,
                'mobile' => $data['mobile'] ?? null,
                'website' => $data['website'] ?? null,
                'address' => $data['address'] ?? null,
                'city' => $data['city'] ?? null,
                'state' => $data['state'] ?? null,
                'postal_code' => $data['postal_code'] ?? null,
                'country' => $data['country'] ?? 'Bahrain',
                'source' => $data['source'] ?? null,
                'status' => $data['status'] ?? 'active',
                'lead_status' => $data['lead_status'] ?? null,
                'cr_number' => $data['cr_number'] ?? null,
                'nationality' => $data['nationality'] ?? null,
                'ownership_type' => $data['ownership_type'] ?? null,
                'business_sector' => $data['business_sector'] ?? null,
                'notes' => $data['notes'] ?? null,
                'custom_fields' => $data['custom_fields'] ?? null,
            ]);

            if (!empty($data['tags'])) {
                $contact->tags()->sync($data['tags']);
            }

            // Log activity
            Activity::log($contact, Activity::TYPE_NOTE, [
                'subject' => 'Contact created',
                'description' => 'Contact was created',
            ]);

            return $contact->fresh(['tags', 'owner']);
        });
    }

    public function update(Contact $contact, array $data): Contact
    {
        return DB::transaction(function () use ($contact, $data) {
            $contact->update($data);

            if (isset($data['tags'])) {
                $contact->tags()->sync($data['tags']);
            }

            return $contact->fresh(['tags', 'owner']);
        });
    }

    public function merge(Contact $primary, array $secondaryIds): Contact
    {
        return DB::transaction(function () use ($primary, $secondaryIds) {
            $secondaries = Contact::whereIn('id', $secondaryIds)->get();

            foreach ($secondaries as $secondary) {
                // Move related records
                $secondary->opportunities()->update(['contact_id' => $primary->id]);
                $secondary->invoices()->update(['contact_id' => $primary->id]);
                $secondary->activities()->update(['contact_id' => $primary->id]);
                $secondary->payments()->update(['contact_id' => $primary->id]);

                // Merge tags
                $primary->tags()->syncWithoutDetaching($secondary->tags->pluck('id'));

                // Fill empty fields from secondary
                foreach ($secondary->getFillable() as $field) {
                    if (empty($primary->$field) && !empty($secondary->$field)) {
                        $primary->$field = $secondary->$field;
                    }
                }

                // Soft delete secondary
                $secondary->delete();
            }

            $primary->save();

            Activity::log($primary, Activity::TYPE_NOTE, [
                'subject' => 'Contacts merged',
                'description' => count($secondaryIds) . ' contact(s) merged into this record',
            ]);

            return $primary->fresh();
        });
    }

    public function import(int $tenantId, UploadedFile $file, array $mapping = []): array
    {
        $results = [
            'created' => 0,
            'updated' => 0,
            'skipped' => 0,
            'errors' => [],
        ];

        $rows = [];
        
        // Parse CSV/Excel
        $extension = $file->getClientOriginalExtension();
        if (in_array($extension, ['csv', 'txt'])) {
            $handle = fopen($file->getPathname(), 'r');
            $headers = fgetcsv($handle);
            while (($row = fgetcsv($handle)) !== false) {
                $rows[] = array_combine($headers, $row);
            }
            fclose($handle);
        }

        foreach ($rows as $index => $row) {
            try {
                $data = $this->mapImportRow($row, $mapping);
                $data['tenant_id'] = $tenantId;

                // Check for existing contact by email
                if (!empty($data['email'])) {
                    $existing = Contact::where('tenant_id', $tenantId)
                        ->where('email', $data['email'])
                        ->first();

                    if ($existing) {
                        $existing->update($data);
                        $results['updated']++;
                        continue;
                    }
                }

                Contact::create($data);
                $results['created']++;
            } catch (\Exception $e) {
                $results['errors'][] = "Row {$index}: " . $e->getMessage();
                $results['skipped']++;
            }
        }

        return $results;
    }

    protected function mapImportRow(array $row, array $mapping): array
    {
        $data = [];
        
        $defaultMapping = [
            'first_name' => ['first_name', 'firstname', 'first name', 'given name'],
            'last_name' => ['last_name', 'lastname', 'last name', 'surname', 'family name'],
            'email' => ['email', 'email address', 'e-mail'],
            'phone' => ['phone', 'phone number', 'telephone', 'tel'],
            'mobile' => ['mobile', 'cell', 'cell phone', 'mobile phone'],
            'company_name' => ['company', 'company_name', 'company name', 'organization'],
            'website' => ['website', 'url', 'web'],
            'address' => ['address', 'street', 'street address'],
            'city' => ['city', 'town'],
            'state' => ['state', 'province', 'region'],
            'postal_code' => ['postal_code', 'zip', 'zip code', 'postcode'],
            'country' => ['country', 'nation'],
        ];

        $mapping = array_merge($defaultMapping, $mapping);

        foreach ($mapping as $field => $possibleNames) {
            foreach ((array) $possibleNames as $name) {
                $lowerName = strtolower($name);
                foreach ($row as $key => $value) {
                    if (strtolower($key) === $lowerName && !empty($value)) {
                        $data[$field] = trim($value);
                        break 2;
                    }
                }
            }
        }

        return $data;
    }

    public function findDuplicates(int $tenantId): array
    {
        $duplicates = [];

        // Find by email
        $emailDuplicates = Contact::where('tenant_id', $tenantId)
            ->whereNotNull('email')
            ->select('email', DB::raw('COUNT(*) as count'))
            ->groupBy('email')
            ->having('count', '>', 1)
            ->get();

        foreach ($emailDuplicates as $dup) {
            $contacts = Contact::where('tenant_id', $tenantId)
                ->where('email', $dup->email)
                ->get();
            
            $duplicates[] = [
                'type' => 'email',
                'value' => $dup->email,
                'contacts' => $contacts,
            ];
        }

        // Find by phone
        $phoneDuplicates = Contact::where('tenant_id', $tenantId)
            ->whereNotNull('phone')
            ->select('phone', DB::raw('COUNT(*) as count'))
            ->groupBy('phone')
            ->having('count', '>', 1)
            ->get();

        foreach ($phoneDuplicates as $dup) {
            $contacts = Contact::where('tenant_id', $tenantId)
                ->where('phone', $dup->phone)
                ->get();
            
            $duplicates[] = [
                'type' => 'phone',
                'value' => $dup->phone,
                'contacts' => $contacts,
            ];
        }

        return $duplicates;
    }

    public function convertLeadToCustomer(Contact $contact): Contact
    {
        $contact->update([
            'type' => 'customer',
            'lead_status' => null,
            'converted_at' => now(),
        ]);

        Activity::log($contact, Activity::TYPE_NOTE, [
            'subject' => 'Lead converted',
            'description' => 'Lead was converted to customer',
        ]);

        return $contact;
    }

    public function getTimeline(Contact $contact, int $limit = 50)
    {
        return Activity::where('contact_id', $contact->id)
            ->with('user')
            ->orderBy('created_at', 'desc')
            ->limit($limit)
            ->get();
    }

    public function addTag(Contact $contact, $tagId): void
    {
        $contact->tags()->syncWithoutDetaching([$tagId]);
    }

    public function removeTag(Contact $contact, $tagId): void
    {
        $contact->tags()->detach($tagId);
    }

    public function calculateLifetimeValue(Contact $contact): float
    {
        return $contact->invoices()
            ->where('status', 'paid')
            ->sum('total');
    }
}
