<?php

namespace App\Services;

use App\Models\Bill;
use App\Models\Contact;
use App\Models\Invoice;
use App\Models\Opportunity;
use App\Models\Payment;
use App\Models\Project;
use App\Models\Task;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class DashboardService
{
    public function getOverview(int $tenantId): array
    {
        $today = now();
        $monthStart = $today->copy()->startOfMonth();
        $monthEnd = $today->copy()->endOfMonth();
        $lastMonthStart = $today->copy()->subMonth()->startOfMonth();
        $lastMonthEnd = $today->copy()->subMonth()->endOfMonth();

        return [
            'revenue' => $this->getRevenueMetrics($tenantId, $monthStart, $monthEnd, $lastMonthStart, $lastMonthEnd),
            'invoices' => $this->getInvoiceMetrics($tenantId),
            'pipeline' => $this->getPipelineMetrics($tenantId),
            'tasks' => $this->getTaskMetrics($tenantId),
            'contacts' => $this->getContactMetrics($tenantId, $monthStart, $monthEnd),
        ];
    }

    protected function getRevenueMetrics(int $tenantId, $monthStart, $monthEnd, $lastMonthStart, $lastMonthEnd): array
    {
        $thisMonth = Payment::where('tenant_id', $tenantId)
            ->whereBetween('payment_date', [$monthStart, $monthEnd])
            ->where('status', 'completed')
            ->sum('amount');

        $lastMonth = Payment::where('tenant_id', $tenantId)
            ->whereBetween('payment_date', [$lastMonthStart, $lastMonthEnd])
            ->where('status', 'completed')
            ->sum('amount');

        $change = $lastMonth > 0 ? (($thisMonth - $lastMonth) / $lastMonth) * 100 : 0;

        return [
            'this_month' => $thisMonth,
            'last_month' => $lastMonth,
            'change_percent' => round($change, 1),
        ];
    }

    protected function getInvoiceMetrics(int $tenantId): array
    {
        $invoices = Invoice::where('tenant_id', $tenantId)
            ->whereNotIn('status', ['draft', 'void'])
            ->selectRaw("
                SUM(total) as total,
                SUM(amount_paid) as paid,
                SUM(CASE WHEN due_date < NOW() AND status != 'paid' THEN total - amount_paid ELSE 0 END) as overdue
            ")
            ->first();

        return [
            'total_outstanding' => ($invoices->total ?? 0) - ($invoices->paid ?? 0),
            'overdue' => $invoices->overdue ?? 0,
            'count' => Invoice::where('tenant_id', $tenantId)
                ->whereNotIn('status', ['draft', 'void', 'paid'])
                ->count(),
        ];
    }

    protected function getPipelineMetrics(int $tenantId): array
    {
        $opportunities = Opportunity::where('tenant_id', $tenantId)
            ->whereNotIn('status', ['won', 'lost'])
            ->get();

        return [
            'total_value' => $opportunities->sum('value'),
            'weighted_value' => $opportunities->sum(fn($o) => $o->value * ($o->probability / 100)),
            'count' => $opportunities->count(),
            'won_this_month' => Opportunity::where('tenant_id', $tenantId)
                ->where('status', 'won')
                ->whereMonth('closed_at', now()->month)
                ->sum('value'),
        ];
    }

    protected function getTaskMetrics(int $tenantId): array
    {
        return [
            'overdue' => Task::where('tenant_id', $tenantId)
                ->where('due_date', '<', now())
                ->whereHas('status', fn($q) => $q->where('is_completed', false))
                ->count(),
            'due_today' => Task::where('tenant_id', $tenantId)
                ->whereDate('due_date', now())
                ->whereHas('status', fn($q) => $q->where('is_completed', false))
                ->count(),
            'in_progress' => Task::where('tenant_id', $tenantId)
                ->whereHas('status', fn($q) => $q->where('name', 'In Progress'))
                ->count(),
            'completed_this_week' => Task::where('tenant_id', $tenantId)
                ->whereBetween('completed_at', [now()->startOfWeek(), now()->endOfWeek()])
                ->count(),
        ];
    }

    protected function getContactMetrics(int $tenantId, $monthStart, $monthEnd): array
    {
        return [
            'total' => Contact::where('tenant_id', $tenantId)->count(),
            'new_this_month' => Contact::where('tenant_id', $tenantId)
                ->whereBetween('created_at', [$monthStart, $monthEnd])
                ->count(),
            'by_source' => Contact::where('tenant_id', $tenantId)
                ->whereNotNull('source')
                ->selectRaw('source, COUNT(*) as count')
                ->groupBy('source')
                ->pluck('count', 'source')
                ->toArray(),
        ];
    }

    public function getRevenueChart(int $tenantId, int $months = 12): array
    {
        $data = [];
        $current = now()->startOfMonth();

        for ($i = $months - 1; $i >= 0; $i--) {
            $month = $current->copy()->subMonths($i);
            $monthStart = $month->copy()->startOfMonth();
            $monthEnd = $month->copy()->endOfMonth();

            $revenue = Payment::where('tenant_id', $tenantId)
                ->whereBetween('payment_date', [$monthStart, $monthEnd])
                ->where('status', 'completed')
                ->sum('amount');

            $expenses = Bill::where('tenant_id', $tenantId)
                ->whereBetween('bill_date', [$monthStart, $monthEnd])
                ->sum('total');

            $data[] = [
                'month' => $month->format('M Y'),
                'revenue' => $revenue,
                'expenses' => $expenses,
                'profit' => $revenue - $expenses,
            ];
        }

        return $data;
    }

    public function getTopCustomers(int $tenantId, int $limit = 5): array
    {
        return Invoice::where('tenant_id', $tenantId)
            ->whereNotIn('status', ['draft', 'void'])
            ->selectRaw('contact_id, SUM(total) as total_revenue')
            ->groupBy('contact_id')
            ->orderByDesc('total_revenue')
            ->limit($limit)
            ->with('contact')
            ->get()
            ->map(fn($row) => [
                'contact' => $row->contact,
                'total_revenue' => $row->total_revenue,
            ])
            ->toArray();
    }

    public function getRecentActivity(int $tenantId, int $limit = 10): array
    {
        $activities = collect();

        // Recent invoices
        $invoices = Invoice::where('tenant_id', $tenantId)
            ->latest()
            ->limit($limit)
            ->get()
            ->map(fn($i) => [
                'type' => 'invoice',
                'title' => "Invoice #{$i->invoice_number} created",
                'amount' => $i->total,
                'date' => $i->created_at,
            ]);

        // Recent payments
        $payments = Payment::where('tenant_id', $tenantId)
            ->latest()
            ->limit($limit)
            ->get()
            ->map(fn($p) => [
                'type' => 'payment',
                'title' => "Payment received",
                'amount' => $p->amount,
                'date' => $p->payment_date,
            ]);

        // Recent opportunities
        $opportunities = Opportunity::where('tenant_id', $tenantId)
            ->latest()
            ->limit($limit)
            ->get()
            ->map(fn($o) => [
                'type' => 'opportunity',
                'title' => "Deal: {$o->name}",
                'amount' => $o->value,
                'date' => $o->created_at,
            ]);

        return $activities
            ->merge($invoices)
            ->merge($payments)
            ->merge($opportunities)
            ->sortByDesc('date')
            ->take($limit)
            ->values()
            ->toArray();
    }
}
