<?php

namespace App\Services;

use App\Models\Department;

class DepartmentService
{
    public function create(int $tenantId, array $data): Department
    {
        return Department::create([
            'tenant_id' => $tenantId,
            'name' => $data['name'],
            'code' => $data['code'] ?? $this->generateCode($data['name']),
            'description' => $data['description'] ?? null,
            'manager_id' => $data['manager_id'] ?? null,
            'parent_id' => $data['parent_id'] ?? null,
        ]);
    }

    public function getHierarchy(int $tenantId): array
    {
        $departments = Department::where('tenant_id', $tenantId)
            ->with(['children', 'manager.user', 'employees'])
            ->whereNull('parent_id')
            ->get();

        return $departments->map(fn($dept) => $this->buildTree($dept))->toArray();
    }

    public function getStats(Department $department): array
    {
        return [
            'employees_count' => $department->employees()->count(),
            'active_employees' => $department->employees()->where('status', 'active')->count(),
            'on_leave' => $department->employees()->where('status', 'on_leave')->count(),
            'budget_used' => $department->employees()
                ->join('employee_salaries', 'employees.id', '=', 'employee_salaries.employee_id')
                ->whereNull('employee_salaries.end_date')
                ->sum(DB::raw('basic_salary + housing_allowance + transport_allowance + other_allowances')),
        ];
    }

    protected function generateCode(string $name): string
    {
        return strtoupper(substr(preg_replace('/[^a-zA-Z]/', '', $name), 0, 4));
    }

    protected function buildTree(Department $department): array
    {
        return [
            'id' => $department->id,
            'name' => $department->name,
            'code' => $department->code,
            'manager' => $department->manager?->user?->full_name,
            'employees_count' => $department->employees->count(),
            'children' => $department->children->map(fn($child) => $this->buildTree($child))->toArray(),
        ];
    }
}
