<?php

namespace App\Services;

use App\Models\Employee;
use App\Models\EmployeeSalary;
use App\Models\LeaveBalance;
use App\Models\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class EmployeeService
{
    public function create(int $tenantId, array $data): Employee
    {
        return DB::transaction(function () use ($tenantId, $data) {
            // Create user account
            $user = User::create([
                'tenant_id' => $tenantId,
                'first_name' => $data['first_name'],
                'last_name' => $data['last_name'],
                'email' => $data['email'],
                'phone' => $data['phone'] ?? null,
                'password' => Hash::make($data['password'] ?? Str::random(12)),
            ]);

            // Create employee record
            $employee = Employee::create([
                'tenant_id' => $tenantId,
                'user_id' => $user->id,
                'employee_id' => $this->generateEmployeeId($tenantId),
                'department_id' => $data['department_id'] ?? null,
                'manager_id' => $data['manager_id'] ?? null,
                'job_title' => $data['job_title'],
                'employment_type' => $data['employment_type'] ?? 'full_time',
                'hire_date' => $data['hire_date'],
                'work_location' => $data['work_location'] ?? null,
                'nationality' => $data['nationality'] ?? null,
                'date_of_birth' => $data['date_of_birth'] ?? null,
                'gender' => $data['gender'] ?? null,
                'marital_status' => $data['marital_status'] ?? null,
                'cpr_number' => $data['cpr_number'] ?? null,
                'passport_number' => $data['passport_number'] ?? null,
                'passport_expiry' => $data['passport_expiry'] ?? null,
                'bank_name' => $data['bank_name'] ?? null,
                'bank_account_number' => $data['bank_account_number'] ?? null,
                'bank_iban' => $data['bank_iban'] ?? null,
                'emergency_contact_name' => $data['emergency_contact_name'] ?? null,
                'emergency_contact_phone' => $data['emergency_contact_phone'] ?? null,
                'emergency_contact_relation' => $data['emergency_contact_relation'] ?? null,
                'status' => 'active',
            ]);

            // Create initial salary record
            if (isset($data['basic_salary'])) {
                EmployeeSalary::create([
                    'tenant_id' => $tenantId,
                    'employee_id' => $employee->id,
                    'basic_salary' => $data['basic_salary'],
                    'housing_allowance' => $data['housing_allowance'] ?? 0,
                    'transport_allowance' => $data['transport_allowance'] ?? 0,
                    'other_allowances' => $data['other_allowances'] ?? 0,
                    'effective_date' => $data['hire_date'],
                ]);
            }

            // Initialize leave balances
            $this->initializeLeaveBalances($employee);

            return $employee;
        });
    }

    public function updateSalary(Employee $employee, array $data): EmployeeSalary
    {
        // End current salary
        $employee->salaries()
            ->whereNull('end_date')
            ->update(['end_date' => now()->subDay()]);

        // Create new salary record
        return EmployeeSalary::create([
            'tenant_id' => $employee->tenant_id,
            'employee_id' => $employee->id,
            'basic_salary' => $data['basic_salary'],
            'housing_allowance' => $data['housing_allowance'] ?? 0,
            'transport_allowance' => $data['transport_allowance'] ?? 0,
            'other_allowances' => $data['other_allowances'] ?? 0,
            'effective_date' => $data['effective_date'] ?? now(),
        ]);
    }

    public function terminate(Employee $employee, array $data): Employee
    {
        $employee->update([
            'status' => 'terminated',
            'termination_date' => $data['termination_date'],
            'termination_reason' => $data['reason'] ?? null,
        ]);

        // End current salary
        $employee->salaries()
            ->whereNull('end_date')
            ->update(['end_date' => $data['termination_date']]);

        return $employee;
    }

    protected function generateEmployeeId(int $tenantId): string
    {
        $count = Employee::where('tenant_id', $tenantId)->count() + 1;
        return 'EMP-' . str_pad($count, 4, '0', STR_PAD_LEFT);
    }

    protected function initializeLeaveBalances(Employee $employee): void
    {
        $leaveTypes = $employee->tenant->leaveTypes;
        
        foreach ($leaveTypes as $leaveType) {
            LeaveBalance::create([
                'tenant_id' => $employee->tenant_id,
                'employee_id' => $employee->id,
                'leave_type_id' => $leaveType->id,
                'year' => now()->year,
                'entitled' => $leaveType->default_days,
                'used' => 0,
                'carried_over' => 0,
            ]);
        }
    }
}
