<?php

namespace App\Services;

use App\Models\Estimate;
use App\Models\EstimateItem;
use Illuminate\Support\Facades\DB;

class EstimateService
{
    public function create(int $tenantId, array $data, int $userId): Estimate
    {
        return DB::transaction(function () use ($tenantId, $data, $userId) {
            $estimate = Estimate::create([
                'tenant_id' => $tenantId,
                'contact_id' => $data['contact_id'],
                'estimate_date' => $data['estimate_date'],
                'expiry_date' => $data['expiry_date'] ?? null,
                'reference' => $data['reference'] ?? null,
                'currency' => $data['currency'] ?? 'BHD',
                'billing_address' => $data['billing_address'] ?? null,
                'notes' => $data['notes'] ?? null,
                'terms' => $data['terms'] ?? null,
                'footer' => $data['footer'] ?? null,
                'discount_type' => $data['discount_type'] ?? null,
                'discount_value' => $data['discount_value'] ?? 0,
                'created_by' => $userId,
            ]);

            $this->syncItems($estimate, $data['items']);
            $estimate->calculateTotals();
            $estimate->save();

            return $estimate;
        });
    }

    public function update(Estimate $estimate, array $data): Estimate
    {
        return DB::transaction(function () use ($estimate, $data) {
            $estimate->update(array_filter([
                'contact_id' => $data['contact_id'] ?? null,
                'estimate_date' => $data['estimate_date'] ?? null,
                'expiry_date' => $data['expiry_date'] ?? null,
                'reference' => $data['reference'] ?? null,
                'currency' => $data['currency'] ?? null,
                'billing_address' => $data['billing_address'] ?? null,
                'notes' => $data['notes'] ?? null,
                'terms' => $data['terms'] ?? null,
                'footer' => $data['footer'] ?? null,
                'discount_type' => $data['discount_type'] ?? null,
                'discount_value' => $data['discount_value'] ?? null,
            ], fn($v) => !is_null($v)));

            if (isset($data['items'])) {
                $this->syncItems($estimate, $data['items']);
            }

            $estimate->calculateTotals();
            $estimate->save();

            return $estimate->fresh();
        });
    }

    protected function syncItems(Estimate $estimate, array $items): void
    {
        $existingIds = [];

        foreach ($items as $index => $item) {
            if (isset($item['id'])) {
                $estimateItem = EstimateItem::find($item['id']);
                $estimateItem->update(array_merge($item, ['position' => $index + 1]));
                $existingIds[] = $item['id'];
            } else {
                $newItem = $estimate->items()->create(array_merge($item, [
                    'tenant_id' => $estimate->tenant_id,
                    'position' => $index + 1,
                ]));
                $existingIds[] = $newItem->id;
            }
        }

        $estimate->items()->whereNotIn('id', $existingIds)->delete();
    }

    public function send(Estimate $estimate, ?string $email = null, ?string $message = null): void
    {
        $recipientEmail = $email ?? $estimate->contact->email;
        // Send email logic here
        $estimate->markAsSent();
    }

    public function duplicate(Estimate $estimate, int $userId): Estimate
    {
        return DB::transaction(function () use ($estimate, $userId) {
            $newEstimate = $estimate->replicate([
                'estimate_number', 'status', 'sent_at', 'viewed_at', 
                'accepted_at', 'declined_at', 'converted_to_invoice_id'
            ]);
            $newEstimate->status = Estimate::STATUS_DRAFT;
            $newEstimate->estimate_date = now();
            $newEstimate->expiry_date = now()->addDays(30);
            $newEstimate->created_by = $userId;
            $newEstimate->save();

            foreach ($estimate->items as $item) {
                $newItem = $item->replicate();
                $newItem->estimate_id = $newEstimate->id;
                $newItem->save();
            }

            return $newEstimate;
        });
    }

    public function generatePdf(Estimate $estimate): string
    {
        // PDF generation logic
        return url('/estimates/' . $estimate->id . '/pdf');
    }
}
