<?php

namespace App\Services;

use App\Models\Expense;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;

class ExpenseService
{
    public function create(int $tenantId, array $data, int $userId): Expense
    {
        $expense = new Expense([
            'tenant_id' => $tenantId,
            'account_id' => $data['account_id'],
            'vendor_id' => $data['vendor_id'] ?? null,
            'contact_id' => $data['contact_id'] ?? null,
            'expense_date' => $data['expense_date'],
            'amount' => $data['amount'],
            'currency' => $data['currency'] ?? 'BHD',
            'tax_rate_id' => $data['tax_rate_id'] ?? null,
            'category' => $data['category'] ?? null,
            'description' => $data['description'] ?? null,
            'is_billable' => $data['is_billable'] ?? false,
            'is_reimbursable' => $data['is_reimbursable'] ?? false,
            'payment_method' => $data['payment_method'] ?? null,
            'reference' => $data['reference'] ?? null,
            'created_by' => $userId,
        ]);

        $expense->calculateTotal();

        if (isset($data['receipt']) && $data['receipt'] instanceof UploadedFile) {
            $expense->receipt_path = $this->uploadReceipt($data['receipt'], $tenantId);
        }

        $expense->save();

        return $expense;
    }

    public function update(Expense $expense, array $data): Expense
    {
        $expense->fill(array_filter([
            'account_id' => $data['account_id'] ?? null,
            'vendor_id' => $data['vendor_id'] ?? null,
            'contact_id' => $data['contact_id'] ?? null,
            'expense_date' => $data['expense_date'] ?? null,
            'amount' => $data['amount'] ?? null,
            'currency' => $data['currency'] ?? null,
            'tax_rate_id' => $data['tax_rate_id'] ?? null,
            'category' => $data['category'] ?? null,
            'description' => $data['description'] ?? null,
            'is_billable' => $data['is_billable'] ?? null,
            'is_reimbursable' => $data['is_reimbursable'] ?? null,
            'payment_method' => $data['payment_method'] ?? null,
            'reference' => $data['reference'] ?? null,
        ], fn($v) => !is_null($v)));

        $expense->calculateTotal();

        if (isset($data['receipt']) && $data['receipt'] instanceof UploadedFile) {
            if ($expense->receipt_path) {
                Storage::delete($expense->receipt_path);
            }
            $expense->receipt_path = $this->uploadReceipt($data['receipt'], $expense->tenant_id);
        }

        $expense->save();

        return $expense;
    }

    protected function uploadReceipt(UploadedFile $file, int $tenantId): string
    {
        return $file->store("tenants/{$tenantId}/receipts", 'local');
    }

    public function approve(Expense $expense, int $userId): void
    {
        $expense->approve(\App\Models\User::find($userId));
    }

    public function reject(Expense $expense, int $userId, ?string $reason = null): void
    {
        $expense->reject(\App\Models\User::find($userId), $reason);
    }
}
