<?php

namespace App\Services;

use App\Models\Contact;
use App\Models\Import;
use Illuminate\Support\Facades\Storage;
use League\Csv\Reader;

class ImportService
{
    public function importContacts(int $tenantId, string $filePath, array $mapping, int $userId): Import
    {
        $import = Import::create([
            'tenant_id' => $tenantId,
            'type' => 'contacts',
            'file_path' => $filePath,
            'mapping' => $mapping,
            'status' => 'processing',
            'created_by' => $userId,
        ]);

        try {
            $csv = Reader::createFromPath(Storage::path($filePath), 'r');
            $csv->setHeaderOffset(0);

            $records = $csv->getRecords();
            $totalRows = 0;
            $successRows = 0;
            $failedRows = 0;
            $errors = [];

            foreach ($records as $index => $record) {
                $totalRows++;

                try {
                    $data = $this->mapRecord($record, $mapping);
                    
                    Contact::updateOrCreate(
                        [
                            'tenant_id' => $tenantId,
                            'email' => $data['email'] ?? null,
                        ],
                        array_merge($data, ['tenant_id' => $tenantId])
                    );

                    $successRows++;
                } catch (\Exception $e) {
                    $failedRows++;
                    $errors[] = [
                        'row' => $index + 2,
                        'error' => $e->getMessage(),
                    ];
                }
            }

            $import->update([
                'status' => 'completed',
                'total_rows' => $totalRows,
                'success_rows' => $successRows,
                'failed_rows' => $failedRows,
                'errors' => $errors,
                'completed_at' => now(),
            ]);
        } catch (\Exception $e) {
            $import->update([
                'status' => 'failed',
                'errors' => [['error' => $e->getMessage()]],
            ]);
        }

        return $import;
    }

    protected function mapRecord(array $record, array $mapping): array
    {
        $data = [];

        foreach ($mapping as $csvColumn => $dbField) {
            if (isset($record[$csvColumn]) && $dbField) {
                $data[$dbField] = trim($record[$csvColumn]);
            }
        }

        return $data;
    }

    public function getPreview(string $filePath, int $rows = 5): array
    {
        $csv = Reader::createFromPath(Storage::path($filePath), 'r');
        $csv->setHeaderOffset(0);

        $headers = $csv->getHeader();
        $records = iterator_to_array($csv->getRecords(), false);
        $preview = array_slice($records, 0, $rows);

        return [
            'headers' => $headers,
            'preview' => $preview,
            'total_rows' => count($records),
        ];
    }

    public function getSuggestedMapping(array $headers): array
    {
        $fieldMap = [
            'first name' => 'first_name',
            'firstname' => 'first_name',
            'last name' => 'last_name',
            'lastname' => 'last_name',
            'email' => 'email',
            'email address' => 'email',
            'phone' => 'phone',
            'phone number' => 'phone',
            'mobile' => 'mobile',
            'company' => 'company_name',
            'company name' => 'company_name',
            'job title' => 'job_title',
            'title' => 'job_title',
            'website' => 'website',
            'address' => 'address_line_1',
            'city' => 'city',
            'state' => 'state',
            'country' => 'country',
            'zip' => 'postal_code',
            'postal code' => 'postal_code',
        ];

        $mapping = [];

        foreach ($headers as $header) {
            $normalized = strtolower(trim($header));
            $mapping[$header] = $fieldMap[$normalized] ?? null;
        }

        return $mapping;
    }
}
