<?php

namespace App\Services;

use App\Models\Notification;
use App\Models\User;
use Illuminate\Support\Facades\DB;

class NotificationService
{
    public function send(User $user, string $type, array $data, ?string $channel = null): Notification
    {
        $notification = Notification::create([
            'tenant_id' => $user->tenant_id,
            'user_id' => $user->id,
            'type' => $type,
            'data' => $data,
            'channel' => $channel ?? 'database',
        ]);

        if ($channel === 'email' || $this->shouldSendEmail($user, $type)) {
            $this->sendEmail($user, $type, $data);
        }

        if ($channel === 'push' || $this->shouldSendPush($user, $type)) {
            $this->sendPush($user, $type, $data);
        }

        return $notification;
    }

    protected function shouldSendEmail(User $user, string $type): bool
    {
        $settings = $user->notification_settings ?? [];
        return $settings['email'][$type] ?? false;
    }

    protected function shouldSendPush(User $user, string $type): bool
    {
        $settings = $user->notification_settings ?? [];
        return $settings['push'][$type] ?? false;
    }

    protected function sendEmail(User $user, string $type, array $data): void
    {
        // Email sending logic
    }

    protected function sendPush(User $user, string $type, array $data): void
    {
        // Push notification logic
    }

    public function getUnread(User $user, int $limit = 20): array
    {
        return Notification::where('user_id', $user->id)
            ->whereNull('read_at')
            ->orderByDesc('created_at')
            ->limit($limit)
            ->get()
            ->toArray();
    }

    public function markAsRead(User $user, ?array $ids = null): int
    {
        $query = Notification::where('user_id', $user->id)->whereNull('read_at');

        if ($ids) {
            $query->whereIn('id', $ids);
        }

        return $query->update(['read_at' => now()]);
    }

    public function markAllAsRead(User $user): int
    {
        return $this->markAsRead($user);
    }

    public function getUnreadCount(User $user): int
    {
        return Notification::where('user_id', $user->id)
            ->whereNull('read_at')
            ->count();
    }

    public function delete(User $user, array $ids): int
    {
        return Notification::where('user_id', $user->id)
            ->whereIn('id', $ids)
            ->delete();
    }

    public function clearAll(User $user): int
    {
        return Notification::where('user_id', $user->id)->delete();
    }
}
