<?php

namespace App\Services;

use App\Events\PayrollProcessed;
use App\Models\Employee;
use App\Models\PayrollItem;
use App\Models\PayrollRun;
use Illuminate\Support\Facades\DB;

class PayrollService
{
    public function createRun(int $tenantId, array $data, int $userId): PayrollRun
    {
        return PayrollRun::create([
            'tenant_id' => $tenantId,
            'pay_period_start' => $data['pay_period_start'],
            'pay_period_end' => $data['pay_period_end'],
            'payment_date' => $data['payment_date'],
            'status' => PayrollRun::STATUS_DRAFT,
            'created_by' => $userId,
        ]);
    }

    public function process(PayrollRun $run): void
    {
        DB::transaction(function () use ($run) {
            $employees = Employee::where('tenant_id', $run->tenant_id)
                ->where('status', 'active')
                ->with('currentSalary')
                ->get();

            $totalGross = 0;
            $totalNet = 0;

            foreach ($employees as $employee) {
                $salary = $employee->currentSalary;
                if (!$salary) continue;

                $item = $this->calculatePayrollItem($run, $employee, $salary);
                $item->save();

                $totalGross += $item->gross_salary;
                $totalNet += $item->net_salary;
            }

            $run->update([
                'status' => PayrollRun::STATUS_PROCESSED,
                'total_gross' => $totalGross,
                'total_net' => $totalNet,
                'processed_at' => now(),
            ]);

            event(new PayrollProcessed($run));
        });
    }

    protected function calculatePayrollItem(PayrollRun $run, Employee $employee, $salary): PayrollItem
    {
        $basicSalary = $salary->basic_salary;
        $housingAllowance = $salary->housing_allowance ?? 0;
        $transportAllowance = $salary->transport_allowance ?? 0;
        $otherAllowances = $salary->other_allowances ?? 0;

        $grossSalary = $basicSalary + $housingAllowance + $transportAllowance + $otherAllowances;

        // Calculate GOSI (Bahrain Social Insurance)
        $gosiEmployee = $this->calculateGosi($employee, $basicSalary, true);
        $gosiEmployer = $this->calculateGosi($employee, $basicSalary, false);

        // Deductions
        $totalDeductions = $gosiEmployee;

        // Net salary
        $netSalary = $grossSalary - $totalDeductions;

        return new PayrollItem([
            'tenant_id' => $run->tenant_id,
            'payroll_run_id' => $run->id,
            'employee_id' => $employee->id,
            'currency' => 'BHD',
            'basic_salary' => $basicSalary,
            'housing_allowance' => $housingAllowance,
            'transport_allowance' => $transportAllowance,
            'other_allowances' => $otherAllowances,
            'gross_salary' => $grossSalary,
            'gosi_employee' => $gosiEmployee,
            'gosi_employer' => $gosiEmployer,
            'tax_amount' => 0, // Bahrain has no income tax
            'other_deductions' => 0,
            'total_deductions' => $totalDeductions,
            'net_salary' => $netSalary,
        ]);
    }

    protected function calculateGosi(Employee $employee, float $basicSalary, bool $isEmployee): float
    {
        // Bahrain GOSI rates:
        // Bahraini: Employee 7%, Employer 12%
        // Non-Bahraini: Employee 1%, Employer 3%
        $isBahraini = strtoupper($employee->nationality) === 'BH';

        if ($isEmployee) {
            return $isBahraini ? $basicSalary * 0.07 : $basicSalary * 0.01;
        }

        return $isBahraini ? $basicSalary * 0.12 : $basicSalary * 0.03;
    }

    public function approve(PayrollRun $run, int $approverId): void
    {
        $run->update([
            'status' => PayrollRun::STATUS_APPROVED,
            'approved_by' => $approverId,
            'approved_at' => now(),
        ]);
    }

    public function markAsPaid(PayrollRun $run): void
    {
        $run->update([
            'status' => PayrollRun::STATUS_PAID,
            'paid_at' => now(),
        ]);

        $run->items()->update(['status' => 'paid']);
    }
}
